/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.objectstore.httphandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.objectstore.ObjectStoreException;

/**
 * Implements the handling of HTTP requests for getting store infos, creating and deleting stores.
 */
public class StoreAdminHandler extends AStoreHandler {

  /** parameter to switch on or off list of objects in store info. */
  public static final String PARAM_RETURNOBJECTS = "returnObjects";

  /** The logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** Default constructor. */
  public StoreAdminHandler() {
  }

  /**
   * Valid methods are GET (store info), PUT (create store) or DELETE (remove store).
   * 
   * {@inheritDoc}
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "PUT".equals(method) || "DELETE".equals(method);
  }

  /**
   * for PUT and DELETE, success status is {@link HttpStatus#CREATED}.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    if ("PUT".equals(method)) {
      return HttpStatus.CREATED;
    } else if ("DELETE".equals(method)) {
      return HttpStatus.OK;
    }
    return super.getSuccessStatus(method, requestUri);
  }

  /**
   * {@inheritDoc}
   * 
   * @return null for create- and delete-store requests, Store information object else.
   */
  @Override
  public Object process(final String method, final String requestUri, final Record parameters)
    throws ObjectStoreException {
    final String storeName = getStoreName(requestUri);
    if ("PUT".equals(method)) {
      return createStore(storeName, parameters);
    } else if ("DELETE".equals(method)) {
      return deleteStore(storeName);
    } else { // if ("GET".equals(method)) {
      return getStoreInfo(storeName, parameters);
    }
  }

  /**
   * create a new store using given configuration properties.
   */
  private Object createStore(final String storeName, final Record parameters) throws ObjectStoreException {
    AnyMap storeProperties = null;
    if (parameters == null) {
      if (_log.isInfoEnabled()) {
        _log.info("Creating store " + storeName + " with default properties.");
      }
    } else {
      storeProperties = parameters.getMetadata();
      if (_log.isInfoEnabled()) {
        _log.info("Creating store " + storeName + " with properties " + storeProperties.toString());
      }
    }
    getObjectStoreService().createStore(storeName, storeProperties);
    return null;
  }

  /**
   * remove store.
   */
  private Object deleteStore(final String storeName) throws ObjectStoreException {
    if (_log.isInfoEnabled()) {
      _log.info("Deleting store " + storeName);
    }
    getObjectStoreService().removeStore(storeName);
    return null;
  }

  /**
   * retrieve store info from {@link org.eclipse.smila.objectstore.ObjectStoreService}.
   */
  private AnyMap getStoreInfo(final String storeName, final Record parameters) throws ObjectStoreException {
    boolean returnObjects = true;
    if (parameters != null && parameters.getMetadata() != null) {
      final Boolean returnObjectsValue = parameters.getMetadata().getBooleanValue(PARAM_RETURNOBJECTS);
      if (returnObjectsValue != null) {
        returnObjects = returnObjectsValue;
      }
    }
    return getObjectStoreService().getStoreInfo(storeName, returnObjects);
  }
}
