/*********************************************************************************************************************
 * Copyright (c) 2008, 2015 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.http.server.jetty;

import java.io.IOException;

import org.eclipse.jetty.security.ConstraintSecurityHandler;
import org.eclipse.jetty.security.MappedLoginService;
import org.eclipse.jetty.server.UserIdentity;
import org.eclipse.jetty.util.security.Credential;
import org.eclipse.jetty.util.security.Password;

/**
 * Simple Jetty LoginService that allows exactly one user/password combination. The password can be given in the
 * password formats usually supported by Jetty: plain text (just the password string), MD5 (prefix "MD5:" plus hash), or
 * encrypted (prefix "CRYPT:" plus encrypted password). You can use the {@link Password} utility to create the hashed or
 * encrypted versions.
 * <p>
 * This LoginService does currently not support definition of user roles, so the security handler must be configured to
 * allow access for "any role" if used with this LoginService (e.g. for the {@link ConstraintSecurityHandler} the
 * "Strict" option must be disabled).
 * <p>
 *
 * Example for usage in jetty.xml:
 *
 * <pre>
 * &lt;Call name="addBean">
 *   &lt;Arg>
 *     &lt;New id="LoginService" class="org.eclipse.smila.http.server.jetty.SingleUserLoginService">
 *       &lt;Set name="name">SMILA Realm&lt;/Set>
 *       &lt;Set name="user">admin&lt;/Set>
 *       &lt;Set name="password">MD5:21232f297a57a5a743894a0e4a801fc3&lt;/Set>
 *     &lt;/New>
 *   &lt;/Arg>
 * &lt;/Call>
 * </pre>
 */
public class SingleUserLoginService extends MappedLoginService {

  private String _userName;

  private String _password;

  /** @return accepted user name. */
  public String getUser() {
    return _userName;
  }

  /** set user name to accept. */
  public void setUser(final String userName) {
    this._userName = userName;
  }

  /** @return password to accept. */
  public String getPassword() {
    return _password;
  }

  /** set password to accept, as plain text, MD5 hash, or encrypted. */
  public void setPassword(final String password) {
    _password = password;
  }

  @Override
  protected void doStart() throws Exception {
    super.doStart();

    final Credential credential = Credential.getCredential(_password);
    putUser(_userName, credential, null); // TODO do we need to define a role?
  }

  @Override
  protected UserIdentity loadUser(final String username) {
    // nothing to do
    return null;
  }

  @Override
  protected void loadUsers() throws IOException {
    // nothing to do
  }

}
