/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.common.definitions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Value;

/**
 * Some job manager related helper methods to deal with Any and their Exception handling.
 * 
 */
public final class AccessAny {

  /**
   * factory for Any.
   */
  public static final DataFactory FACTORY = DataFactory.DEFAULT;

  /** prevent instance creation. */
  private AccessAny() {
  }

  /**
   * get a field value from a Map and check that it has an expected type.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @param fieldType
   *          expected field type
   * @return the value for the key or null if it doesn't exist.
   * @throws InvalidDefinitionException
   *           any is not a map, value does not have expected type or error accessing any.
   */
  public static Any get(final AnyMap any, final String fieldName, final Any.ValueType fieldType)
    throws InvalidDefinitionException {
    Any field = null;
    field = any.get(fieldName);
    if (field != null && field.getValueType() != fieldType) {
      throw new InvalidDefinitionException("Field " + fieldName + " does not have type " + fieldType);
    }
    return field;
  }

  /**
   * get a value expression from a Map.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the value expression for the key or null if it doesn't exist.
   * @throws InvalidDefinitionException
   *           any is not a map, value is not a string or error accessing any.
   */
  public static ValueExpression getValueExpression(final AnyMap any, final String fieldName)
    throws InvalidDefinitionException {
    final String expression = getString(any, fieldName);
    if (expression != null) {
      return new ValueExpression(expression);
    }
    return null;
  }

  /**
   * get a string from a Map.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the string for the key or null if it doesn't exist.
   * @throws InvalidDefinitionException
   *           any is not a map, value is not a string or error accessing any.
   */
  public static String getString(final AnyMap any, final String fieldName) throws InvalidDefinitionException {
    final Value field = (Value) get(any, fieldName, ValueType.STRING);
    if (field != null) {
      return field.asString();
    }
    return null;
  }

  /**
   * get a sequence from a Map.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the sequence for the key or null if it doesn't exist.
   * @throws InvalidDefinitionException
   *           any is not a map, value is not a sequence or error accessing any.
   */
  public static AnySeq getSeq(final AnyMap any, final String fieldName) throws InvalidDefinitionException {
    final Any field = get(any, fieldName, ValueType.SEQ);
    if (field != null) {
      return field.asSeq();
    }
    return null;
  }

  /**
   * get a boolean from a Map. Accept the string values "true" or "false", too.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @param defaultValue
   *          value to return if key is not set.
   * @return the boolean for the key
   * @throws InvalidDefinitionException
   *           any is not a map, value is not a boolean or error accessing any.
   */
  public static boolean getBoolean(final AnyMap any, final String fieldName, final boolean defaultValue)
    throws InvalidDefinitionException {
    final Any field = any.get(fieldName);
    if (field == null) {
      return defaultValue;
    }
    if (field.isValue()) {
      try {
        return field.asValue().asBoolean();
      } catch (final InvalidValueTypeException ex) {
        throw new InvalidDefinitionException("Field " + fieldName + " does not have a boolean value.", ex);
      }
    }
    throw new InvalidDefinitionException("Field " + fieldName + " does not have a boolean value.");
  }

  /**
   * like {@link #getValueExpression(Any, String)}, but throw an exception if the field does not exists.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the value expression for the key.
   * @throws InvalidDefinitionException
   *           any is not a map, value does not exists or is not a string or error accessing any.
   */
  public static ValueExpression getValueExpressionRequired(final AnyMap any, final String fieldName)
    throws InvalidDefinitionException {
    return new ValueExpression(getStringRequired(any, fieldName));
  }

  /**
   * like {@link #getString(Any, String)}, but throw an exception if the field does not exists.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the string for the key.
   * @throws InvalidDefinitionException
   *           any is not a map, value does not exists or is not a string or error accessing any.
   */
  public static String getStringRequired(final AnyMap any, final String fieldName)
    throws InvalidDefinitionException {
    final String value = getString(any, fieldName);
    if (StringUtils.isEmpty(value)) {
      throw new InvalidDefinitionException("Missing field '" + fieldName + "' or it has empty value.");
    }
    return value;
  }

  /**
   * like {@link #getStringRequired(Any, String)}, but throw an exception if value is not a valid name accoring to
   * {@link NameValidator}.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the name value for the key.
   * @throws InvalidDefinitionException
   *           any is not a map, value does not exists or is not a string or not valid or error accessing any.
   */
  public static String getValidName(final AnyMap any, final String fieldName) throws InvalidDefinitionException {
    final String name = getStringRequired(any, fieldName);
    try {
      NameValidator.checkName(name);
    } catch (final InvalidDefinitionException ex) {
      throw new InvalidDefinitionException("Value '" + name + "' in field '" + fieldName + "' is not valid: "
        + ex.getMessage());
    }
    return name;
  }

  /**
   * get list of strings from a field of the any.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the list of strings or null if the field does not exists.
   * @throws InvalidDefinitionException
   *           any is not a map, value is not an sequence of strings or error accessing any.
   */
  public static List<String> getStringSeq(final AnyMap any, final String fieldName)
    throws InvalidDefinitionException {
    final AnySeq field = (AnySeq) get(any, fieldName, ValueType.SEQ);
    List<String> values = null;
    if (field != null) {
      values = new ArrayList<String>();
      try {
        for (final Any element : field) {
          if (!element.isString()) {
            throw new InvalidDefinitionException("Sequence in field " + fieldName + " must contain strings only");
          }
          values.add(((Value) element).asString());
        }
      } catch (final InvalidDefinitionException ex) {
        throw ex;
      }
    }
    return values;
  }

  /**
   * Clones AnyMap object, support null arguments, too.
   * 
   * @param map
   *          an AnyMap object or null
   * @return clone of map, or null
   */
  public static AnyMap cloneAnyMap(final AnyMap map) {
    if (map == null) {
      return null;
    }
    return map.getFactory().cloneAnyMap(map);
  }

  /**
   * Create list of elements of given collection object, support null arguments, too.
   * 
   * @param map
   *          a collection or null
   * @return list containing the elements of input collection.
   */
  public static <T> List<T> copyCollection(final Collection<T> collection) {
    if (collection == null) {
      return null;
    }
    return new ArrayList<T>(collection);
  }

  /**
   * Create shallow copy of given map, support null arguments, too.
   * 
   * @param map
   *          a map or null
   * @return map containing all key/values pairs from input map.
   */
  public static <K, V> Map<K, V> copyMap(final Map<K, V> map) {
    if (map == null) {
      return null;
    }
    return new HashMap<K, V>(map);
  }
}
