/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.httphandler;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import org.codehaus.jackson.JsonParseException;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.bulkbuilder.InvalidJobException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.validation.InvalidRecordException;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.WorkflowRunInfo;

/**
 * Implements the processing of HTTP requests for the bulk processing.
 */
public class BulkbuilderHandler extends JsonRequestHandler {

  /** result key for the url to get infos about a workflow run. */
  public static final String KEY_URL = "url";

  /** Prefix for the URL in the response to get infos about a specific workflow run. */
  public static final String JOB_URL_PREFIX = "/jobmanager/jobs";

  /** Reference to the BulkBuilder. */
  protected BulkbuilderService _bulkbuilderService;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord,
    final HttpExchange exchange) throws Exception {
    final String jobName = getJobName(requestUri);
    WorkflowRunInfo workflowRunInfo = null;
    if ("POST".equals(method)) {
      if (inputRecord != null) {
        workflowRunInfo = _bulkbuilderService.addRecord(jobName, inputRecord);
      } else {
        workflowRunInfo = _bulkbuilderService.commitJob(jobName);
      }
    } else if ("DELETE".equals(method)) {
      if (inputRecord != null) {
        workflowRunInfo = _bulkbuilderService.deleteRecord(jobName, inputRecord);
      } else {
        workflowRunInfo = _bulkbuilderService.commitJob(jobName);
      }
    }
    if (workflowRunInfo != null) {
      return createResponse(workflowRunInfo, getRequestHost(exchange), getRootContextPath());
    }
    return null;
  }

  /**
   * create Any object for client response.
   */
  public static Any createResponse(final WorkflowRunInfo workflowRunInfo, final String requestHost,
    final String rootContext) throws MalformedURLException {
    final AnyMap result = workflowRunInfo.toAny();
    final String workflowRunPath =
      workflowRunInfo.getJobName() + "/" + workflowRunInfo.getJobId() + "/workflowrun/"
        + workflowRunInfo.getWorkflowId();
    final URL workflowRunUrl = URLCreator.create(requestHost, rootContext + JOB_URL_PREFIX, workflowRunPath);
    result.put(KEY_URL, workflowRunUrl.toString());
    return result;
  }

  /**
   * @param requestUri
   *          the request URI
   * @return the job name extracted from the request URI.
   */
  protected String getJobName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * @return true if method is either POST or DELETE.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "POST".equals(method) || "DELETE".equals(method);
  }

  /**
   * {@inheritDoc}
   * 
   * @return ACCEPTED
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    return HttpStatus.ACCEPTED;
  }

  /**
   * {@inheritDoc}
   * 
   * @return HTTP error code matching the exception type.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof InvalidRecordException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex instanceof InvalidValueTypeException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex instanceof JsonParseException || ex.getCause() != null && ex.getCause() instanceof JsonParseException) {
      return HttpStatus.BAD_REQUEST;
    }
    if (ex instanceof InvalidJobException) {
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof BulkbuilderException) {
      return HttpStatus.SERVICE_UNAVAILABLE;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * Sets the bulkbuilder.
   * 
   * @param service
   *          the new bulkbuilder
   */
  public void setBulkbuilderService(final BulkbuilderService service) {
    _bulkbuilderService = service;
  }

  /**
   * Unset the bulkbuilder.
   * 
   * @param service
   *          the bulkbuilder to unset
   */
  public void unsetBulkbuilderService(final BulkbuilderService service) {
    if (_bulkbuilderService == service) {
      _bulkbuilderService = null;
    }
  }
}
