/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.outputs;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.taskworker.output.AppendableOutput;

/** record output for appendable record bulks. */
public class AppendableBulkOutput extends BulkOutput {
  /** local Logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** create instance. */
  public AppendableBulkOutput(final String jobName, final String bulkId, final BulkType bulkType) {
    super(jobName, bulkId, bulkType);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addRecord(final Record record) throws BulkbuilderException {
    try {
      ((AppendableOutput) getOutput()).appendRecord(record);
    } catch (final IOException ex) {
      throw new BulkbuilderException(ex.getClass().getSimpleName() + " writing record to bulk "
        + getCurrentBulkId(), ex);
    } catch (final ObjectStoreException ex) {
      throw new BulkbuilderException(ex.getClass().getSimpleName() + " writing record to bulk "
        + getCurrentBulkId(), ex);
    }
  }

  /** put a micro bulk to the bulk. */
  public void addMicroBulk(final byte[] microBulk, final Integer numberOfRecords) throws BulkbuilderException {
    try {
      ((AppendableOutput) getOutput()).append(microBulk, numberOfRecords);
    } catch (final ObjectStoreException ex) {
      throw new BulkbuilderException(ex.getClass().getSimpleName() + " writing record to bulk "
        + getCurrentBulkId(), ex);
    }
  }

  @Override
  public void commit() {
    try {
      super.commit();
    } catch (final BulkbuilderException ex) {
      _log.info("Appendable bulk could not be finished, but this is not critical.", ex);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getBytesWritten() {
    return ((AppendableOutput) getOutput()).getBytesWritten();
  }

  /**
   * @return <code>true</code> if the bulk has timed out (if a timeout is set).
   */
  @Override
  public boolean isTimedOut() {
    return getCommitTimeout() >= 0 && System.currentTimeMillis() - getBulkStartTime() >= getCommitTimeout();
  }

}
