/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client;

import org.apache.http.HttpStatus;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.http.client.impl.DefaultRestClient;
import org.eclipse.smila.utils.MaybeRecoverableException;

/**
 * Exception thrown by {@link DefaultRestClient} to wrap JSON errors returned by the REST API. The HTTP status code can
 * be read from the exception. If the request returns a JSON error object, it will be contained in the exception. If the
 * object has a key "message", the value will be the message of this exception. For status codes equal to or greater
 * than 500 (INTERNAL SERVER ERROR) the exception will be flagged as {@link #isRecoverable()} because such status codes
 * often describe only temporary issues and it is possible that a retry of the request will succeed.
 */
public class RestException extends MaybeRecoverableException {
  private static final long serialVersionUID = 1L;

  private static final String KEY_MESSAGE = "message";

  private final int _responseCode;

  private final AnyMap _responseObject;

  /** Create an instance with a message when no response object is available. */
  public RestException(final int responseCode, final String message) {
    super(message, isRecoverable(responseCode));
    _responseCode = responseCode;
    _responseObject = null;
  }

  /** Create instance from response object. */
  public RestException(final int responseCode, final AnyMap responseObject) {
    super(getMessage(responseObject), isRecoverable(responseCode));
    _responseCode = responseCode;
    _responseObject = responseObject;
  }

  /** Determine if response status code is 500 or greater. */
  private static boolean isRecoverable(final int responseCode) {
    return responseCode >= HttpStatus.SC_INTERNAL_SERVER_ERROR;
  }

  /** Get the message from the response object, if possible. */
  private static String getMessage(final AnyMap responseObject) {
    if (responseObject != null) {
      final Any message = responseObject.get(KEY_MESSAGE);
      if (message != null && message.isString()) {
        return message.asValue().asString();
      }
    }
    return "Response did not contain a message.";
  }

  /**
   * Get the HTTP response code of the failed request. See {@link HttpStatus} for possible values.
   * 
   * @return HTTP response status code.
   */
  public int getResponseCode() {
    return _responseCode;
  }

  /**
   * Get the original JSON object of the response. May be null.
   * 
   * @return JSON response object if available.
   */
  public AnyMap getResponseObject() {
    return _responseObject;
  }
}
