/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.JobDependencyException;

/**
 * Implements the handling of HTTP requests to cancel a job workflow run.
 * 
 * URL pattern: smila/jobmanager/jobs/-jobname-/-jobrunid-/workflowrun/-workflowrunid-/-command-
 */
public class WorkflowRunControlHandler extends AJobManagerHandler {

  /** command value to invoke a job workflow run cancel operation. */
  public static final String CMD_CANCEL = "cancel";

  /** valid number of dynamic URI parts. */
  private static final int URIPARTS_COUNT = 4;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() == URIPARTS_COUNT) { // TODO
      final String jobName = uriParts.get(0);
      final String jobRunId = uriParts.get(1);
      final String workflowRunId = uriParts.get(2);
      final String command = uriParts.get(3);
      if (CMD_CANCEL.equals(command)) {
        getJobRunEngine().cancelWorkflowRun(jobName, jobRunId, workflowRunId);
      } else {
        throw new IllegalArgumentException("Invalid request URI, operation '" + command
          + "' not supported for job runs");
      }
      return null;
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'");
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalJobStateException caused by a RunStorageException to the
   * exception handling of {@link AJobManagerHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalJobStateException) {
      return ((IllegalJobStateException) ex).isJobRunGone() ? HttpStatus.GONE : HttpStatus.BAD_REQUEST;
    } else if (ex instanceof JobDependencyException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @return the status code to return for successful requests using the method and request URI. The code is "ACCEPTED"
   *         (202).
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    return HttpStatus.OK;
  }

  /**
   * {@inheritDoc}
   * 
   * POST is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return isPostRequest(method);
  }
}
