/**********************************************************************************************************************
 * Copyright (c) 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This program
 * and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.search.api.helper;

import java.util.List;

import org.apache.commons.lang.NullArgumentException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.search.api.SearchResultConstants;

/**
 * wrapper helper class to build the result record. This is intentionally a subclass of the {@link ResultAccessor} as to
 * also provide read access to the props that are set here.
 */
public class ResultBuilder extends ResultAccessor {

  public ResultBuilder(final Record result) {
    super(result);

    _result.getMetadata().getSeq(SearchResultConstants.RECORDS, true);
  }

  /**
   * Adds a new result item to the list. the given args are mandatory and will be set at the item.
   *
   * @return the map containing the id and score/weight under their proper names.
   */
  public AnyMap addResultItem(final String id, final Double score) {
    final AnyMap resultItem = getResultRecords().getFactory().createAnyMap();
    getResultRecords().add(resultItem);
    resultItem.put(Record.RECORD_ID, id);
    resultItem.put(SearchResultConstants.WEIGHT, score);
    return resultItem;
  }

  /**
   * Add highlight text to result item.
   *
   * @param resultItem
   *          the {@link AnyMap} result item.
   * @param attribute
   *          the highlighted attribute.
   * @param text
   *          the highlight text.
   */
  public void addHighlightText(final AnyMap resultItem, final String attribute, final String text) {
    if (text == null) {
      throw new NullArgumentException("text");
    }
    final AnyMap highlight = _result.getFactory().createAnyMap();
    highlight.put(SearchResultConstants.HIGHLIGHT_TEXT, text);
    addHighlight(resultItem, attribute, highlight);
  }

  /**
   * Add multiple highlight texts to result item.
   *
   * @param resultItem
   *          the {@link AnyMap} result item.
   * @param attribute
   *          the highlighted attribute.
   * @param text
   *          a {@link List} containing highlight texts.
   */
  public void addHighlightText(final AnyMap resultItem, final String attribute, final List<String> text) {
    if (text == null) {
      throw new NullArgumentException("text");
    }
    final AnyMap highlight = _result.getFactory().createAnyMap();
    highlight.getSeq(SearchResultConstants.HIGHLIGHT_TEXT, true).addAll(AnyUtil.objectToAny(text).asSeq());
    addHighlight(resultItem, attribute, highlight);
  }

  /**
   * Add highlight result to result item.
   *
   * @param resultItem
   *          the {@link AnyMap} result item.
   * @param attribute
   *          the highlighted attribute.
   * @param highlight
   *          the highlight result.
   */
  public void addHighlight(final AnyMap resultItem, final String attribute, final AnyMap highlight) {
    if (attribute == null) {
      throw new NullArgumentException("attribute");
    }
    if (highlight == null) {
      throw new NullArgumentException("highlight");
    }
    resultItem.getMap(SearchResultConstants.HIGHLIGHT, true).put(attribute, highlight);
  }

  /**
   * sets value for {@link SearchResultConstants#COUNT}.
   */
  public void setCount(final Long count) {
    _result.getMetadata().put(SearchResultConstants.COUNT, count);
  }

  /**
   * sets value for {@link SearchResultConstants#RUNTIME}.
   */
  public void setRuntime(final Long millis) {
    _result.getMetadata().put(SearchResultConstants.RUNTIME, millis);
  }

  /**
   * Adds facets Seq with the given name and returns it. The facet map is created on the fly if needs be.
   */
  public AnySeq addFacet(final String facetName) {
    final AnyMap facets = _result.getMetadata().getMap(SearchResultConstants.FACETS, true);

    return facets.getSeq(facetName, true);
  }

  /**
   * Adds the map containing the count and value for the facet value.
   *
   * @param facet
   *          NullArgumentException if null
   * @param value
   *          NullArgumentException if null
   * @param count
   *          if null the count property is not set
   * @return the map of the added facet
   */
  public AnyMap addFacetValue(final AnySeq facet, final Any value, final Long count) {
    if (facet == null) {
      throw new NullArgumentException("facet");
    }
    if (value == null) {
      throw new NullArgumentException("value");
    }

    final AnyMap facetValue = facet.getFactory().createAnyMap();
    facetValue.put(SearchResultConstants.VALUE, value);
    if (count != null) {
      facetValue.put(SearchResultConstants.COUNT, count);
    }
    facet.add(facetValue);
    return facetValue;
  }

  /**
   * Add a map containing facet value and count to facet.
   *
   * @param facet
   *          facet to which value is added.
   * @param value
   *          value to add.
   * @param count
   *          count of facet value.
   * @return map containing value and count.
   */
  public AnyMap addFacetValue(final AnySeq facet, final Object value, final Long count) {
    final Value autoConvertValue = facet.getFactory().autoConvertValue(value);
    return addFacetValue(facet, autoConvertValue, count);
  }

  /**
   * Add a map containing facet value with specified {@link ValueType} and count to facet.
   *
   * @param facet
   *          facet to which value is added.
   * @param value
   *          value to add.
   * @param valueType
   *          type of value.
   * @param count
   *          count of facet value.
   * @return map containing value and count.
   */
  public AnyMap addFacetValue(final AnySeq facet, final String value, final ValueType valueType, final Long count) {
    final Value autoConvertValue = facet.getFactory().parseFromString(value, valueType);
    return addFacetValue(facet, autoConvertValue, count);
  }

  public AnySeq addGroup(final String name) {
    return addGroup(name, null);
  }

  public AnySeq addGroup(final String name, final Long matches) {
    if (name == null) {
      throw new NullArgumentException("name");
    }
    final AnyMap groups = _result.getMetadata().getMap(SearchResultConstants.GROUPS, true);
    if (matches != null) {
      final DataFactory factory = groups.getFactory();
      groups.put(SearchResultConstants.MATCHES, factory.autoConvertValue(matches));
    }
    return groups.getSeq(name, true);
  }

  /**
   * Add a group to the {@link AnySeq} group with value, count and results.
   *
   * @param group
   *          the group to add to.
   * @param value
   *          the value.
   * @param count
   *          the count.
   * @param results
   *          the results.
   * @return the added group as {@link AnyMap}.
   */
  public AnyMap addGroupResults(final AnySeq group, final Any value, final Long count, final AnySeq results) {
    final AnyMap map = addToGroup(group, value, count);
    map.put(SearchResultConstants.RESULTS, results);
    return map;
  }

  /**
   * Add a group to the {@link AnySeq} group with value, count and a nested group.
   *
   * @param group
   *          the group to add to.
   * @param value
   *          the value.
   * @param count
   *          the count.
   * @param nestedGroup
   *          the nested group.
   * @return the added group as {@link AnyMap}.
   */
  public AnyMap addNestedGroup(final AnySeq group, final Any value, final Long count, final AnyMap nestedGroup) {
    final AnyMap map = addToGroup(group, value, count);
    map.put(SearchResultConstants.GROUPS, nestedGroup);
    return map;
  }

  /**
   * Create an {@link AnyMap} with value and count and add it the the given group.
   *
   * @param group
   *          the group to add to.
   * @param value
   *          the value.
   * @param count
   *          the count.
   * @return the added group as {@link AnyMap}.
   */
  private AnyMap addToGroup(final AnySeq group, final Any value, final Long count) {
    if (group == null) {
      throw new NullArgumentException("group");
    }
    if (value == null) {
      throw new NullArgumentException("value");
    }
    final AnyMap map = group.getFactory().createAnyMap();
    map.put(SearchResultConstants.VALUE, value);
    if (count != null) {
      map.put(SearchResultConstants.COUNT, count);
    }
    group.add(map);
    return map;
  }

}
