/***********************************************************************************************************************
 * Copyright (c) 2008,2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 **********************************************************************************************************************/
package org.eclipse.smila.importing.worker;

import java.util.Collection;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.DeltaException;
import org.eclipse.smila.importing.DeltaImportStrategy;
import org.eclipse.smila.importing.DeltaService;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.State;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;

/**
 * base class for worker accessing a {@link DeltaService}. Provides service reference handling and methods to invoke the
 * {@link DeltaService} methods and measure the durations of the calls in the task context counters.
 */
public abstract class WorkerUsingDeltaService implements Worker {
  /** Reference to DeltaService. */
  private DeltaService _deltaService;

  /** Log. */
  private final Log _log = LogFactory.getLog(getClass());

  /**
   * get job run id from task.
   * 
   * @throws IllegalArgumentException
   *           if not set.
   */
  protected String getJobRunId(final TaskContext taskContext) {
    final String jobRunId = taskContext.getTask().getProperties().get(Task.PROPERTY_JOB_RUN_ID);
    if (jobRunId == null) {
      throw new IllegalArgumentException("Missing property '" + Task.PROPERTY_JOB_RUN_ID + "' in task: "
        + taskContext.getTask());
    }
    return jobRunId;
  }

  /**
   * get a string parameter.
   * 
   * @throws IllegalArgumentException
   *           if not set.
   */
  protected String getRequiredParameter(final TaskContext taskContext, final String parameter) {
    final String targetJobName = taskContext.getTaskParameters().getStringValue(parameter);
    if (targetJobName == null) {
      throw new IllegalArgumentException("Missing parameter '" + parameter + "' in task: " + taskContext.getTask());
    }
    return targetJobName;
  }

  /**
   * get deltaImportStrategy parameter from task context. Fall back to {@link DeltaImportStrategy#FULL} if parameter is
   * not set or value is invalid.
   */
  protected DeltaImportStrategy getDeltaImportStrategy(final TaskContext taskContext) {
    return getDeltaImportStrategy(taskContext.getTaskParameters());
  }

  /**
   * get deltaImportStrategy parameter from task parameters. Fall back to {@link DeltaImportStrategy#FULL} if parameter
   * is not set or value is invalid.
   */
  protected static DeltaImportStrategy getDeltaImportStrategy(final AnyMap taskParameters) {
    DeltaImportStrategy usage = DeltaImportStrategy.FULL;
    final String paramValue = taskParameters.getStringValue(DeltaImportStrategy.TASK_PARAM);
    if (paramValue != null) {
      try {
        usage = DeltaImportStrategy.valueOf(paramValue.toUpperCase(Locale.ENGLISH));
      } catch (final Exception ex) {
        ; // ignore.
      }
    }
    return usage;
  }

  /** get value of attribute {@link ImportingConstants#ATTRIBUTE_COMPOUNDID}. */
  protected String getCompoundId(final Record record) {
    return record.getMetadata().getStringValue(ImportingConstants.ATTRIBUTE_COMPOUNDID);
  }

  /** check if attribute {@link ImportingConstants#ATTRIBUTE_COMPOUNDFLAG} is set to true. */
  protected boolean isCompound(final Record record) {
    if (record.getMetadata().containsKey(ImportingConstants.ATTRIBUTE_COMPOUNDFLAG)) {
      return record.getMetadata().getBooleanValue(ImportingConstants.ATTRIBUTE_COMPOUNDFLAG);
    }
    return false;
  }

  /**
   * invoke {@link DeltaService#checkState(String, String, String, String)} and measure time as duration
   * <tt>checkDeltaState</tt>.
   */
  public State checkDeltaStateTimed(final String jobRunId, final Record record, final String deltaHash,
    final TaskContext taskContext) throws DeltaException {
    final long startTime = taskContext.getTimestamp();
    try {
      final String compoundId = getCompoundId(record);
      final State deltaStatus =
        _deltaService.checkState(record.getSource(), record.getId(), compoundId, jobRunId, deltaHash);
      if (deltaStatus == State.CHANGED) {
        record.getMetadata().put(ImportingConstants.ATTRIBUTE_UPDATE, true);
      }
      return deltaStatus;
    } catch (final DeltaException ex) {
      // treat record as update, if deltaservice fails to check. Don't fail because of this.
      _log.warn("Error checking state of record " + record.getId() + ", treating as update.", ex);
      record.getMetadata().put(ImportingConstants.ATTRIBUTE_UPDATE, true);
      return State.CHANGED;
    } finally {
      taskContext.measureTime("checkDeltaState", startTime);
    }
  }

  /**
   * invoke {@link DeltaService#markCompoundElementsVisited(String, String, String)} and measure time as duration
   * <tt>markCompoundElementsVisited</tt>.
   */
  public void markCompoundElementsVisitedTimed(final String jobRunId, final Record record,
    final TaskContext taskContext) throws DeltaException {
    final long startTime = taskContext.getTimestamp();
    try {
      _deltaService.markCompoundElementsVisited(record.getSource(), record.getId(), jobRunId);
    } catch (final DeltaException ex) {
      // don't fail because of this.
      _log.warn("Error marking compound element " + record.getId() + " as updated.", ex);
    } finally {
      taskContext.measureTime("markCompoundElementsVisited", startTime);
    }
  }

  /**
   * invoke {@link DeltaService#markAsUpdated(String, String, String, String)} and measure time as duration
   * <tt>markAsUpdated</tt>.
   */
  public void markAsUpdatedTimed(final String jobRunId, final Record record, final String deltaHash,
    final TaskContext taskContext) throws DeltaException {
    final long startTime = taskContext.getTimestamp();
    try {
      final String compoundId = getCompoundId(record);
      _deltaService.markAsUpdated(record.getSource(), record.getId(), compoundId, jobRunId, deltaHash);
    } catch (final DeltaException ex) {
      // don't fail because of this.
      _log.warn("Error marking record " + record.getId() + " as updated.", ex);
    } finally {
      taskContext.measureTime("markAsUpdated", startTime);
    }
  }

  /**
   * invoke {@link DeltaService#getUnvisitedEntries(String, String)} and measure time as duration
   * <tt>getUnvisitedRecords</tt>.
   */
  public Collection<DeltaService.EntryId> getUnvisitedEntriesTimed(final String jobRunId,
    final String sourceAndShardPrefix, final TaskContext taskContext) throws DeltaException {
    final long startTime = taskContext.getTimestamp();
    try {
      return _deltaService.getUnvisitedEntries(sourceAndShardPrefix, jobRunId);
    } finally {
      taskContext.measureTime("getUnvisitedRecords", startTime);
    }
  }

  /**
   * invoke {@link DeltaService#deleteEntry(String, String)} and measure time as duration <tt>deleteDeltaEntry</tt>.
   */
  public void deleteDeltaEntryTimed(final String sourceId, final DeltaService.EntryId entryId,
    final TaskContext taskContext) throws DeltaException {
    final long startTime = taskContext.getTimestamp();
    try {
      _deltaService.deleteEntry(sourceId, entryId);
    } finally {
      taskContext.measureTime("deleteDeltaEntry", startTime);
    }
  }

  /** DS service reference bind method. */
  public void setDeltaService(final DeltaService service) {
    _deltaService = service;
  }

  /** DS service reference unbind method. */
  public void unsetDeltaService(final DeltaService service) {
    if (_deltaService == service) {
      _deltaService = null;
    }
  }

}
