/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.restapi;

import java.util.Map;

import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.taskmanager.TaskCounter;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.versions.VersionManager;

/**
 * Implements the handling of requests to /smila:
 *
 * <pre>
 * {
 *   "taskmanager" : {
 *     "todo" : 0,
 *    "inprogress" : 0
 *   },
 *   "api" : {
 *     "jobmanager" : "http://localhost:8080/smila/jobmanager/",
 *     "tasks" : "http://localhost:8080/smila/tasks/",
 *     ...
 *   }
 * }
 * </pre>
 */
public class StateHandler extends JsonRequestHandler {

  /** the taskmanager to use for getting task counters. */
  private TaskManager _taskManager;

  /** the versionmanager to use for get the SMILA version. */
  private VersionManager _versionManager;

  /** the cluster config service to use for get the cluster noces. */
  private ClusterConfigService _clusterConfig;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord,
    final HttpExchange exchange) {
    final AnyMap result = FACTORY.createAnyMap();
    result.put("version", _versionManager.getVersionInfos());
    result.put("cluster", buildClusterSection());
    result.put("taskmanager", buildTaskManagerSection());
    result.put("api", buildApiSection(getRequestHost(exchange), requestUri));
    return result;
  }

  /**
   * @return AnyMap, representing cluster section
   */
  private AnyMap buildClusterSection() {
    try {
      final AnyMap clusterSection = FACTORY.createAnyMap();
      final AnySeq nodes = FACTORY.createAnySeq();
      for (String node : _clusterConfig.getClusterNodes()) {
        nodes.add(node);
      }
      clusterSection.put("nodes", nodes);
      return clusterSection;
    } catch (final Exception e) {
      return buildErrorSection(e);
    }
  }

  /**
   * build taskmanager section from task counters.
   *
   * @return Any, representing taskmanager section
   */
  private AnyMap buildTaskManagerSection() {
    final AnyMap taskManagerSection = FACTORY.createAnyMap();
    try {
      int todoCount = 0;
      int inprogressCount = 0;
      final Map<String, TaskCounter> counters = _taskManager.getTaskCounters();
      for (final TaskCounter counter : counters.values()) {
        todoCount += counter.getTasksTodo();
        inprogressCount += counter.getTasksInProgress();
      }
      taskManagerSection.put("todo", todoCount);
      taskManagerSection.put("inprogress", inprogressCount);
      return taskManagerSection;
    } catch (final Exception e) {
      return buildErrorSection(e);
    }
  }

  /**
   * Build "api" section.
   *
   * @return Any, representing api section
   */
  private AnyMap buildApiSection(final String requestHost, final String requestUri) {
    final AnyMap section = FACTORY.createAnyMap();
    try {
      section.put("jobmanager", URLCreator.create(requestHost, requestUri, "jobmanager").toString());
      section.put("tasks", URLCreator.create(requestHost, requestUri, "tasks").toString());
      section.put("pipelines", URLCreator.create(requestHost, requestUri, "pipeline").toString());
      section.put("pipelets", URLCreator.create(requestHost, requestUri, "pipelets").toString());
      section.put("store", URLCreator.create(requestHost, requestUri, "store").toString());
      section.put("importing", URLCreator.create(requestHost, requestUri, "importing").toString());
      section.put("scripts", URLCreator.create(requestHost, requestUri, "script").toString());
      section.put("solr", URLCreator.create(requestHost, "solr", "admin").toString());
    } catch (final Exception e) {
      section.put("error", e.getClass().getCanonicalName() + ": " + e.getMessage());
    }
    return section;
  }

  /** convert exception to AnyMap. */
  private AnyMap buildErrorSection(final Exception e) {
    final AnyMap errorSection = FACTORY.createAnyMap();
    errorSection.put("error", AnyUtil.exceptionToAny(e));
    return errorSection;
  }

  /**
   * {@inheritDoc}
   *
   * GET is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /** set OSGI service. */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /** unset OSGI service. */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (taskManager == _taskManager) {
      _taskManager = null;
    }
  }

  /** set OSGI service. */
  public void setVersionManager(final VersionManager versionManager) {
    _versionManager = versionManager;
  }

  /** unset OSGI service. */
  public void unsetVersionManager(final VersionManager versionManager) {
    if (versionManager == _versionManager) {
      _versionManager = null;
    }
  }

  /** set OSGI service. */
  public void setClusterConfigService(final ClusterConfigService clusterConfig) {
    _clusterConfig = clusterConfig;
  }

  /** unset OSGI service. */
  public void unsetClusterConfigService(final ClusterConfigService clusterConfig) {
    if (clusterConfig == _clusterConfig) {
      _clusterConfig = null;
    }
  }

}
