/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.PipeletTracker;
import org.eclipse.smila.scripting.ScriptExecutor;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.commonjs.module.ModuleScriptProvider;
import org.osgi.service.component.ComponentContext;

/**
 * Implements {@link org.eclipse.smila.scripting.ScriptingEngine} by using Mozilla Rhino to evaluate Javascript files.
 */
public class JavascriptEngine implements ScriptingEngine {

  public static final String BUNDLE_NAME = "org.eclipse.smila.scripting";

  private static final String SYSPROP_SCRIPT_DIR = "smila.scripting.dir";

  private static final String SYSPROP_SCRIPT_DEBUG = "smila.scripting.debug";

  private static final String DEFAULT_SCRIPT_DIR = "js";

  private Path _scriptDirectory;

  private ModuleScriptProvider _scriptProvider;

  private Scriptable _baseScope;

  private PipeletTracker _pipeletTracker;

  private BlackboardFactory _blackboardFactory;

  private final Log _log = LogFactory.getLog(getClass());

  protected void activate(final ComponentContext componentContext) throws IOException {
    RhinoUtils.printEngineInfo(_log);

    setScriptDirectory();
    initDebugger();

    _scriptProvider = RhinoUtils.createScriptProvider(_scriptDirectory);
    _baseScope =
      RhinoUtils.createBaseScope(componentContext.getBundleContext(), _pipeletTracker, _blackboardFactory);
  }

  private void setScriptDirectory() {
    final String scriptDirProperty = System.getProperty(SYSPROP_SCRIPT_DIR);
    if (scriptDirProperty == null) {
      final String configFolder = ConfigUtils.getConfigurationFolder().getAbsolutePath();
      _scriptDirectory = Paths.get(configFolder, BUNDLE_NAME, DEFAULT_SCRIPT_DIR);
    } else {
      _scriptDirectory = Paths.get(scriptDirProperty);
    }
    _scriptDirectory = _scriptDirectory.toAbsolutePath();
    _log.info("Using Javascript files from directory " + _scriptDirectory);
  }

  private void initDebugger() {
    final String connectionString = System.getProperty(SYSPROP_SCRIPT_DEBUG);
    if (!StringUtils.isEmpty(connectionString)) {
      try {
        RhinoUtils.startDebugger(connectionString);
        _log.info("RhinoDebugger initialised with settings " + connectionString);
      } catch (final Exception ex) {
        _log.warn("Failed to start debugger", ex);
      }
    }
  }

  @Override
  public Collection<String> listScriptNames() throws ScriptingEngineException {
    final AnySeq scripts = listScripts();
    final Set<String> scriptNames = new HashSet<>();
    for (final Any script : scripts) {
      scriptNames.add(script.asMap().getStringValue("name"));
    }
    return scriptNames;
  }

  @Override
  public AnySeq listScripts() throws ScriptingEngineException {
    final AnySeq scripts = DataFactory.DEFAULT.createAnySeq();
    if (!Files.exists(_scriptDirectory) || !Files.isReadable(_scriptDirectory)
      || !Files.isDirectory(_scriptDirectory)) {
      _log.warn("Script directory " + _scriptDirectory
        + " does not exists, is not readable or not a directory. Cannot list scripts.");
    } else {
      try (ScriptExecutor executor = getScriptExecutor()) {
        Files.walkFileTree(_scriptDirectory, new ScriptCatalogVisitor(executor, scripts, _scriptDirectory));
      } catch (final IOException ex) {
        throw new ScriptingEngineException("Error reading script catalog files", ex);
      }
    }
    sortDescriptionsByName(scripts);
    return scripts;
  }

  private void sortDescriptionsByName(final AnySeq scripts) {
    Collections.sort(scripts, new Comparator<Any>() {
      @Override
      public int compare(final Any o1, final Any o2) {
        if (o1.isMap() && o2.isMap()) {
          return o1.asMap().getStringValue("name").compareTo(o2.asMap().getStringValue("name"));
        }
        return 0;
      };
    });
  }

  @Override
  public Record callScript(final String scriptName, final Record record) throws ScriptingEngineException {
    final String[] scriptFileAndFunction = getScriptFileAndFunction(scriptName);
    try (final ScriptExecutor executor = getScriptExecutor()) {
      executor.loadScript(scriptFileAndFunction[0]);
      final String scriptFunction = scriptFileAndFunction[1];
      return executor.call(scriptFunction, record);
    }
  }

  @Override
  public AnyMap callScript(final String scriptName, final AnyMap arguments) throws ScriptingEngineException {
    final String[] scriptFileAndFunction = getScriptFileAndFunction(scriptName);
    try (final ScriptExecutor executor = getScriptExecutor()) {
      executor.loadScript(scriptFileAndFunction[0]);
      final String scriptFunction = scriptFileAndFunction[1];
      return executor.call(scriptFunction, arguments);
    }
  }

  private String[] getScriptFileAndFunction(final String scriptName) throws ScriptingEngineException {
    final int dotIndex = scriptName.indexOf('.');
    if (dotIndex < 0) {
      throw new ScriptingEngineException("Script name " + scriptName
        + " does not contain a dot, must have format <script>.<function>");
    }
    return new String[] { scriptName.substring(0, dotIndex), scriptName.substring(dotIndex + 1) };
  }

  @Override
  public ScriptExecutor getScriptExecutor() throws ScriptingEngineException {
    return new JavascriptExecutor(_scriptProvider, _baseScope);
  }

  /** used by DS to set reference to pipelet tracker. */
  public void setPipeletTracker(final PipeletTracker pipeletTracker) {
    _pipeletTracker = pipeletTracker;
  }

  /** used by DS to remove reference to pipelet tracker. */
  public void unsetPipeletTracker(final PipeletTracker pipeletTracker) {
    if (_pipeletTracker == pipeletTracker) {
      _pipeletTracker = null;
    }
  }

  /** set blackboard factory reference (used by DS). */
  public void setBlackboardFactory(final BlackboardFactory factory) {
    _blackboardFactory = factory;
  }

  /** remove blackboard factory reference (used by DS). */
  public void unsetBlackboardFactory(final BlackboardFactory factory) {
    if (_blackboardFactory == factory) {
      _blackboardFactory = null;
    }
  }

}
