/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.UUID;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.mozilla.javascript.Scriptable;

/**
 * Wrapper for executing pipelets in scripts. Allows to call Pipelet.process() without Blackboard parameter.
 */
public class PipeletWrapper {

  private final Pipelet _pipelet;

  private final BlackboardFactory _blackboardFactory;

  private final Scriptable _baseScope;

  /** Creates the wrapper. */
  public PipeletWrapper(final Pipelet pipelet, final BlackboardFactory blackboardFactory, final Scriptable baseScope) {
    _pipelet = pipelet;
    _blackboardFactory = blackboardFactory;
    _baseScope = baseScope;
  }

  /**
   * Process the input object with the wrapped pipelet. The input object is converted (or unwrapped) into one or more
   * records and added to a transient blackboard. The pipelet is invoked with this blackboard and the IDs of the
   * records. The records for the returned IDs are extracted from the blackboard and wrapped as an array for further use
   * in Javascript.
   *
   * @param input
   *          input object, representing one or more records.
   * @return an array of {@link RecordWrapper}.
   * @throws ScriptingEngineException
   *           error during record processing.
   */
  public RecordWrapper[] process(final Object input) throws ScriptingEngineException {
    final Record[] records = RecordWrapper.asRecordArray(input);
    final Record[] result = process(records);
    if (result != null) {
      final RecordWrapper[] wrappedResult = new RecordWrapper[result.length];
      for (int i = 0; i < result.length; i++) {
        wrappedResult[i] = new RecordWrapper(result[i], _baseScope);
      }
      return wrappedResult;
    }
    return null;
  }

  /**
   * Process given records with wrapped pipelet: Put them on a blackboard, run the pipelet, extract the result records
   * from the blackboard.
   *
   * @param records
   *          records to process
   * @return result records
   * @throws ScriptingEngineException
   *           error during record processing.
   */
  private Record[] process(final Record[] records) throws ScriptingEngineException {
    final Blackboard blackboard = _blackboardFactory.createTransientBlackboard();
    final Collection<String> ids = new ArrayList<>();
    try {
      for (final Record record : records) {
        if (record.getId() == null) {
          record.setId(UUID.randomUUID().toString());
        }
        blackboard.setRecord(record);
        ids.add(record.getId());
      }
      final String[] recordIds = ids.toArray(new String[0]);
      final String[] resultIds = _pipelet.process(blackboard, recordIds);
      final Collection<Record> result = new ArrayList<>();
      for (final String id : resultIds) {
        result.add(blackboard.getRecord(id));
      }
      return result.toArray(new Record[0]);
    } catch (ProcessingException | BlackboardAccessException | RuntimeException e) {
      throw new ScriptingEngineException("Error while calling Pipelet " + _pipelet.getClass().getSimpleName(), e);
    } finally {
      blackboard.unload();
    }
  }
}
