/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.Wrapper;

/**
 * Wraps a SMILA record as a native Javascript object with these properties.
 *
 * Attributes of the record can be accessed like Javascript properties, just like in a AnyMapWrapper. To access the
 * attachments, use special property {@value RecordWrapper#PROPERTY_ATTACHMENTS}. The
 * {@value RecordWrapper#PROPERTY_ATTACHMENTS} property itself cannot be changed or deleted.
 *
 * The special property {@value RecordWrapper#PROPERTY_METADATA} can be used to get an AnyMapWrapper for the metadata of
 * this record. This may be necessary to call Java methods that expect an AnyMap argument and therefore cannot be
 * invoked with the record itself. The {@value RecordWrapper#PROPERTY_METADATA} property itself cannot be changed or
 * deleted.
 *
 * Finally, the special property {@value RecordWrapper#PROPERTY_ID} can be used to get and set the record ID. This is
 * equivalent to using property "_recordid".
 */
public class RecordWrapper extends ObjectWrapperBase {

  private static final String PROPERTY_ID = "$id";

  private static final String PROPERTY_METADATA = "$metadata";

  private static final String PROPERTY_ATTACHMENTS = "$attachments";

  private final Record _record;

  private final AnyMapWrapper _metadata;

  private final AttachmentWrapper _attachments;

  /** Wraps a record for access from Javascripts. */
  public RecordWrapper(final Record record, final Scriptable parentScope) {
    super(parentScope);
    _record = record;
    _metadata = new AnyMapWrapper(_record.getMetadata(), parentScope);
    _attachments = new AttachmentWrapper(_record, parentScope);
  }

  /** Unwrap or convert the given object to be used as a Record. */
  public static Record asRecord(Object input) {
    if (input instanceof Wrapper) {
      input = ((Wrapper) input).unwrap();
    }
    if (input instanceof Record) {
      return (Record) input;
    } else if (input instanceof Map) {
      final AnyMap anyMap = (AnyMap) AnyUtil.objectToAny(input);
      final Record record = DataFactory.DEFAULT.createRecord(anyMap);
      if (record.getId() == null) {
        record.setId(UUID.randomUUID().toString());
      }
      return record;
    } else {
      throw new IllegalArgumentException("Cannot convert object to record: " + input);
    }
  }

  /** Unwrap or convert the given object to be used as a Record[]. */
  public static Record[] asRecordArray(Object input) {
    if (input instanceof Wrapper) {
      input = ((Wrapper) input).unwrap();
    }
    if (input instanceof Object[]) {
      input = Arrays.asList((Object[]) input);
    }
    if (input instanceof List) {
      final List<?> inputList = (List<?>) input;
      final Record[] records = new Record[inputList.size()];
      for (int i = 0; i < inputList.size(); i++) {
        records[i] = asRecord(inputList.get(i));
      }
      return records;
    } else {
      return new Record[] { asRecord(input) };
    }

  }

  @Override
  public Record unwrap() {
    return _record;
  }

  @Override
  public String getClassName() {
    return "RecordWrapper";
  }

  @Override
  public Object get(final String name, final Scriptable start) {
    switch (name) {
      case PROPERTY_ID:
        return _record.getId();
      case PROPERTY_METADATA:
        return _metadata;
      case PROPERTY_ATTACHMENTS:
        return _attachments;
      default:
        return _metadata.get(name, start);
    }
  }

  @Override
  public Object get(final int index, final Scriptable start) {
    return _metadata.get(index, start);
  }

  @Override
  public void put(final String name, final Scriptable start, final Object value) {
    switch (name) {
      case PROPERTY_ID:
        _record.setId(value.toString());
        break;
      case PROPERTY_METADATA:
      case PROPERTY_ATTACHMENTS:
        Context.reportRuntimeError("Cannot set property " + name);
        break;
      default:
        _metadata.put(name, start, value);
    }
  }

  @Override
  public void put(final int index, final Scriptable start, final Object value) {
    _metadata.put(index, start, value);
  }

  @Override
  public void delete(final String name) {
    switch (name) {
      case PROPERTY_ID:
      case PROPERTY_METADATA:
      case PROPERTY_ATTACHMENTS:
        Context.reportRuntimeError("Cannot delete property " + name);
        break;
      default:
        _metadata.delete(name);
    }
  }

  @Override
  public void delete(final int index) {
    _metadata.delete(index);
  }

  @Override
  public boolean has(final String name, final Scriptable start) {
    return _metadata.has(name, start);
  }

  @Override
  public boolean has(final int index, final Scriptable start) {
    return _metadata.has(index, start);
  }

  @Override
  public Object[] getIds() {
    return _metadata.getIds();
  }

}
