/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.worker;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.scripting.ScriptExecutor;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.eclipse.smila.scripting.internal.AnyWrapper;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.output.RecordOutput;
import org.mozilla.javascript.BaseFunction;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.ScriptRuntime;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;
import org.mozilla.javascript.Undefined;
import org.mozilla.javascript.Wrapper;

/**
 * Implementation of emit() function available in scripts used by {@link ScriptProcessorWorker} to write records into
 * the output bulk.
 */
public class EmitFunction extends BaseFunction implements ScriptExecutor.Installable {
  private static final long serialVersionUID = 1L;

  private final RecordOutput _outputBulk;

  @SuppressWarnings("unused")
  private final TaskContext _taskContext;

  public EmitFunction(final RecordOutput outputBulk, final TaskContext taskContext) {
    _outputBulk = outputBulk;
    _taskContext = taskContext;
  }

  /**
   * Binds this instance of emit() into the specified scope under the property name "emit".
   * 
   * @param installTarget
   *          the scope where the emit() function is to be installed, must implement {@link Scriptable}.
   */
  @Override
  public void install(final Object installTarget) throws ScriptingEngineException {
    if (installTarget instanceof Scriptable) {
      final Scriptable scope = (Scriptable) installTarget;
      ScriptableObject.putProperty(scope, getFunctionName(), this);
    } else {
      throw new ScriptingEngineException("Cannot install emit() function in object of type "
        + installTarget.getClass());
    }
  }

  @Override
  public Object call(final Context cx, final Scriptable scope, final Scriptable thisObj, final Object[] args) {
    if (args == null || args.length != 1) {
      throw ScriptRuntime.throwError(cx, scope, "emit() needs one argument");
    }
    Object argument = args[0];
    if (argument instanceof Wrapper) {
      argument = ((Wrapper) argument).unwrap();
    }
    try {
      if (argument instanceof Record) {
        _outputBulk.writeRecord((Record) argument);
      } else {
        final Any object = AnyWrapper.unwrapAny(argument);
        if (object != null) {
          _outputBulk.writeAny(object);
        }
      }
    } catch (final Exception ex) {
      throw Context.throwAsScriptRuntimeEx(ex);
    }
    return Undefined.instance;
  }

  @Override
  public Scriptable construct(final Context cx, final Scriptable scope, final Object[] args) {
    throw ScriptRuntime.throwError(cx, scope, "emit() can not be invoked as a constructor");
  }

  @Override
  public String getFunctionName() {
    return "emit";
  }

  @Override
  public int getArity() {
    return 1;
  }

  @Override
  public int getLength() {
    return 1;
  }
}
