/***********************************************************************************************************************
 * Copyright (c) 2008,2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 **********************************************************************************************************************/
package org.eclipse.smila.importing.worker;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.importing.DeltaException;
import org.eclipse.smila.importing.DeltaImportStrategy;
import org.eclipse.smila.importing.DeltaService;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.jobmanager.taskgenerator.DefaultTaskGenerator;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorException;
import org.eclipse.smila.taskmanager.Task;

/**
 * task generator for {@link UpdatePusherWorker}:
 * <ul>
 * <li>assert that either "jobToPushTo" or the "remote" section is set.
 * <li>create one completing task per shard in delta service, if delta-delete is enabled.
 * </ul>
 */
public class UpdatePusherTaskGenerator extends DefaultTaskGenerator {
  private DeltaService _deltaService;

  /** logger. */
  private final Log _log = LogFactory.getLog(getClass());

  @Override
  public void validateParameters(final AnyMap parameters) throws TaskGeneratorException {
    super.validateParameters(parameters);
    // TODO use constants from UpdatePusherWorker.
    if (parameters.containsKey("jobToPushTo") == parameters.containsKey("remote")) {
      throw new TaskGeneratorException(
        "UpdatePusher worker requires exactly one of the 'jobToPushTo' or 'remote' parameters.");
    }
  }

  @Override
  public List<Task> createCompletionTasks(final Map<String, Bucket> inputBuckets,
    final Map<String, Bucket> outputBuckets, final AnyMap parameters, final String workerName)
    throws TaskGeneratorException {
    final List<Task> tasks = new ArrayList<Task>();
    final DeltaImportStrategy deltaUsage = WorkerUsingDeltaService.getDeltaImportStrategy(parameters);
    if (deltaUsage.isDeltaDeleteEnabled()) {
      final String sourceId = parameters.getStringValue(ImportingConstants.TASK_PARAM_DATA_SOURCE);
      try {
        final Collection<String> shards = _deltaService.getShardPrefixes(sourceId);
        for (final String shard : shards) {
          final Task task = createCompletionTask(parameters, workerName);
          task.getParameters().put(UpdatePusherWorker.DELETE_SHARD_PARAM, shard);
          tasks.add(task);
        }
      } catch (final DeltaException ex) {
        throw new TaskGeneratorException("Failed to create completion tasks", ex);
      }
    } else {
      _log.info("Delta-Delete is disabled for usage = " + deltaUsage + ", no tasks created.");
    }
    return tasks;
  }

  /** DS service reference bind method. */
  public void setDeltaService(final DeltaService deltaService) {
    _deltaService = deltaService;
  }

  /** DS service reference unbind method. */
  public void unsetDeltaService(final DeltaService deltaService) {
    if (_deltaService == deltaService) {
      _deltaService = null;
    }
  }

}
