/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.file.filter.FilterConfiguration;
import org.eclipse.smila.importing.crawler.file.filter.FilterEvaluator;
import org.eclipse.smila.importing.util.CrawlingContext;
import org.eclipse.smila.taskworker.TaskContext;

/** contains all parameters needed to control the crawl process. */
public final class FileCrawlingContext extends CrawlingContext {

  private final String _rootFolder;

  private final long _maxFilesPerBulk;

  private final long _minFilesPerBulk;

  private final long _directoriesPerBulk;

  private final FilterConfiguration _filterConfig;

  private final FilterEvaluator _filterEvaluator;

  public FileCrawlingContext(final TaskContext taskContext) {
    super(taskContext, false);
    final AnyMap taskParameters = taskContext.getTaskParameters();
    _rootFolder = taskParameters.getStringValue(FileCrawlerWorker.TASK_PARAM_ROOT_FOLDER);
    if (_rootFolder == null || _rootFolder.trim().length() == 0) {
      throw new IllegalArgumentException("Parameter '" + FileCrawlerWorker.TASK_PARAM_ROOT_FOLDER + "' of task "
        + taskContext.getTask().getTaskId() + " is null or empty");
    }
    _maxFilesPerBulk =
      getNonNegativeParameter(taskParameters, FileCrawlerWorker.TASK_PARAM_MAX_FILES_PER_BULK,
        FileCrawlerWorker.MAX_FILES_PER_BULK_DEFAULT);
    if (_maxFilesPerBulk == 0) {
      throw new IllegalArgumentException("Task parameter " + FileCrawlerWorker.TASK_PARAM_MAX_FILES_PER_BULK
        + " must be greater than 0");
    }
    _minFilesPerBulk =
      getNonNegativeParameter(taskParameters, FileCrawlerWorker.TASK_PARAM_MIN_FILES_PER_BULK,
        FileCrawlerWorker.MIN_FILES_PER_BULK_DEFAULT);
    if (_maxFilesPerBulk <= _minFilesPerBulk) {
      throw new IllegalArgumentException("Task parameter " + FileCrawlerWorker.TASK_PARAM_MAX_FILES_PER_BULK
        + " is less or equal to " + FileCrawlerWorker.TASK_PARAM_MIN_FILES_PER_BULK);
    }
    _directoriesPerBulk =
      getNonNegativeParameter(taskParameters, FileCrawlerWorker.TASK_PARAM_DIRS_PER_BULK,
        FileCrawlerWorker.DIRS_PER_BULK_DEFAULT);
    if (taskParameters.get(ImportingConstants.TASK_PARAM_FILTERS) != null) {
      _filterConfig = new FilterConfiguration(taskParameters.getMap(ImportingConstants.TASK_PARAM_FILTERS));
    } else {
      _filterConfig = null;
    }
    _filterEvaluator = new FilterEvaluator();
    _filterEvaluator.setFilterConfiguration(_filterConfig);
  }

  public String getRootFolder() {
    return _rootFolder;
  }

  public FilterEvaluator getFilterEvaluator() {
    return _filterEvaluator;
  }

  public FilterConfiguration getFilterConfig() {
    return _filterConfig;
  }

  public long getMaxFilesPerBulk() {
    return _maxFilesPerBulk;
  }

  public long getMinFilesPerBulk() {
    return _minFilesPerBulk;
  }

  public long getDirectoriesPerBulk() {
    return _directoriesPerBulk;
  }
}
