/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file.internal;

import static java.nio.file.attribute.AclEntryPermission.READ_DATA;
import static java.nio.file.attribute.AclEntryPermission.WRITE_DATA;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PRINCIPAL_OTHERS;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PROPERTY_FILE_EXTENSION;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PROPERTY_FILE_FOLDER;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PROPERTY_FILE_LAST_MODIFIED;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PROPERTY_FILE_NAME;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PROPERTY_FILE_PATH;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PROPERTY_FILE_READ_ACL;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PROPERTY_FILE_SIZE;
import static org.eclipse.smila.importing.crawler.file.FileCrawlerService.PROPERTY_FILE_WRITE_ACL;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.AclEntry;
import java.nio.file.attribute.AclFileAttributeView;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.crawler.file.FileCrawlerService;
import org.eclipse.smila.importing.util.FilePathNormalizer;

/**
 * Utility class to create records from File objects.
 * 
 * @author stuc07
 * 
 */
public final class FileToRecordConverter {

  private final Log _log = LogFactory.getLog(getClass());

  private final DataFactory _factory;

  public FileToRecordConverter(final DataFactory factory) {
    _factory = factory;
  }

  /**
   * Create a Record for the given file and dataSource and fill it with all file properties. Optionally also fill with
   * content attachment.
   * 
   * @param file
   * @param dataSource
   * @param withContent
   *          if true fill also content attachment, otherwise not
   * @return a Record
   * @throws IOException
   */
  public Record fileToRecord(final Path file, final String dataSource, final boolean withContent)
    throws IOException {
    if (Files.isDirectory(file)) {
      throw new IllegalArgumentException("parameter file is a directory");
    }

    final Record record = createRecord(file, dataSource);
    addFileProperties(file, record);
    record.getMetadata().put(ImportingConstants.ATTRIBUTE_DELTA_HASH,
      Long.toString(Files.getLastModifiedTime(file).toMillis()));
    if (withContent) {
      addAttachment(record, FileCrawlerService.PROPERTY_FILE_PATH, FileCrawlerService.ATTACHMENT_FILE_CONTENT);
    }
    return record;
  }

  /**
   * Create a Record for the given directory and dataSource and fill it with ImportingConstants.ROOT_FOLDER_ATTRIBUTE.
   * 
   * @param directory
   * @param dataSource
   * @return a record
   * @throws IOException
   */
  public Record dirToRecord(final Path directory, final String dataSource) throws IOException {
    if (!Files.isDirectory(directory)) {
      throw new IllegalArgumentException("parameter directory is not a directory");
    }
    final Record record = createRecord(directory, dataSource);
    record.getMetadata().put(FileCrawlerService.PROPERTY_FILE_FOLDER, directory.toRealPath().toString());
    return record;
  }

  /**
   * Add the content attachment to the given record. The file content is expected to be referenced by PROPERTY_FILE_PATH
   * in the record.
   * 
   * @param record
   * @param pathAttribute
   *          name of the attribute containing the path to the file
   * @param attachmentName
   *          name with which the attachment is stored in the record
   * @throws IOException
   */
  public void addAttachment(final Record record, final String pathAttribute, final String attachmentName)
    throws IOException {
    final String path = record.getMetadata().getStringValue(pathAttribute);
    if (path == null) {
      throw new IllegalArgumentException("Record '" + record.getId() + "' does not contain attribute '"
        + pathAttribute + "'");
    }
    record.setAttachment(attachmentName, Files.readAllBytes(Paths.get(path)));
  }

  private Record createRecord(final Path file, final String dataSource) throws IOException {
    return _factory.createRecord(dataSource + ":" + file.toRealPath(), dataSource);
  }

  private void addFileProperties(final Path file, final Record record) throws IOException {
    final AnyMap metadata = record.getMetadata();
    metadata.put(PROPERTY_FILE_NAME, file.getFileName().toString());
    metadata.put(PROPERTY_FILE_PATH, FilePathNormalizer.getNormalizedPath(file.toRealPath()));
    if (file.getParent() != null) {
      metadata.put(PROPERTY_FILE_FOLDER, FilePathNormalizer.getNormalizedPath(file.getParent()));
    }
    metadata.put(PROPERTY_FILE_SIZE, Files.size(file));
    final Value lastModifiedDate =
      _factory.createDateTimeValue(new Date(Files.getLastModifiedTime(file).toMillis()));
    metadata.put(PROPERTY_FILE_LAST_MODIFIED, lastModifiedDate);
    metadata.put(PROPERTY_FILE_EXTENSION, FilenameUtils.getExtension(file.getFileName().toString()));
    final boolean haveAclPermissions = getAclFilePermissions(file, metadata);
    if (!haveAclPermissions) {
      getPosixFilePermissions(file, metadata);
    }
  }

  private boolean getAclFilePermissions(final Path file, final AnyMap metadata) {
    try {
      final AclFileAttributeView aclAttributeView = Files.getFileAttributeView(file, AclFileAttributeView.class);
      if (aclAttributeView != null) {
        final List<AclEntry> accessControlList = aclAttributeView.getAcl();
        if (accessControlList != null) {
          for (final AclEntry ace : accessControlList) {
            if (ace.permissions().contains(READ_DATA)) {
              metadata.getSeq(PROPERTY_FILE_READ_ACL, true).add(ace.principal().getName());
            }
            if (ace.permissions().contains(WRITE_DATA)) {
              metadata.getSeq(PROPERTY_FILE_WRITE_ACL, true).add(ace.principal().getName());
            }
          }
          return true;
        }
      }
    } catch (final Exception e) {
      _log.warn("Couldn't fetch access control list on file " + file, e);
    }
    return false;
  }

  private boolean getPosixFilePermissions(final Path file, final AnyMap metadata) {
    try {
      final PosixFileAttributeView posixAttributeView =
        Files.getFileAttributeView(file, PosixFileAttributeView.class);
      if (posixAttributeView != null) {
        final PosixFileAttributes posixFileAttributes = posixAttributeView.readAttributes();
        final Set<PosixFilePermission> permissions = posixFileAttributes.permissions();
        if (permissions != null) {
          final String owner = posixFileAttributes.owner().getName();
          final String group = posixFileAttributes.group().getName();
          for (final PosixFilePermission permission : permissions) {
            switch (permission) {
              case OWNER_READ:
                metadata.getSeq(PROPERTY_FILE_READ_ACL, true).add(owner);
                break;
              case OWNER_WRITE:
                metadata.getSeq(PROPERTY_FILE_WRITE_ACL, true).add(owner);
                break;
              case GROUP_READ:
                metadata.getSeq(PROPERTY_FILE_READ_ACL, true).add(group);
                break;
              case GROUP_WRITE:
                metadata.getSeq(PROPERTY_FILE_WRITE_ACL, true).add(group);
                break;
              case OTHERS_READ:
                metadata.getSeq(PROPERTY_FILE_READ_ACL, true).add(PRINCIPAL_OTHERS);
                break;
              case OTHERS_WRITE:
                metadata.getSeq(PROPERTY_FILE_WRITE_ACL, true).add(PRINCIPAL_OTHERS);
                break;
              default:
                break;
            }
          }
          return true;
        }
      }
    } catch (final Exception e) {
      _log.warn("Couldn't fetch POSIX file attributes on file " + file, e);
    }
    return false;
  }
}
