/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.persistence.objectstore;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.TreeSet;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.ipc.IpcAnyWriter;
import org.eclipse.smila.datamodel.ipc.IpcRecordReader;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.JobDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkflowDefinition;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.StoreOutputStream;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.osgi.service.component.ComponentContext;

/**
 * Component for handling persistent jobmanager data by using objectstore.
 */
public class PermanentStorageObjectstore implements org.eclipse.smila.jobmanager.persistence.PermanentStorage {

  /** the store to persist the jobmanager data. */
  private static final String JOBMANAGER_STORE = "jobmanager";

  /** the prefix for bucket data. */
  private static final String BUCKETS_PREFIX = "buckets/";

  /** the prefix for job data. */
  private static final String JOBS_PREFIX = "jobs/";

  /** the prefix for workflow data. */
  private static final String WORKFLOWS_PREFIX = "workflows/";

  /** the prefix for job run data. */
  private static final String JOB_RUNS_PREFIX = "runs/";

  /** max retries for store requests on IOExceptions. */
  private static final int MAX_RETRY_ON_STORE_UNAVAILABLE = 3;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** objectStore service. */
  private ObjectStoreService _objectStoreService;

  /** for BON/JSON -> Record parsing. */
  private final IpcRecordReader _recordReader = new IpcRecordReader();

  /** for Any->(pretty)JSON serialization. (We want pretty JSON for logging) */
  private final IpcAnyWriter _anyWriter = new IpcAnyWriter(true);

  /** has store already been successfully prepared? */
  private boolean _isStorePrepared;

  /**
   * OSGi Declarative Services service activation method.
   * 
   * @param context
   *          OSGi service component context.
   * @throws ObjectStoreException
   *           could not prepare store.
   */
  protected void activate(final ComponentContext context) throws ObjectStoreException {
    if (_log.isDebugEnabled()) {
      _log.debug("activate");
    }
  }

  /**
   * OSGi Declarative Services service deactivation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void deactivate(final ComponentContext context) {
    if (_log.isDebugEnabled()) {
      _log.debug("deactivate");
    }
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param objectStoreService
   *          ObjectStoreService reference.
   */
  public void setObjectStoreService(final ObjectStoreService objectStoreService) {
    _objectStoreService = objectStoreService;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param objectStoreService
   *          ObjectStoreService reference.
   */
  public void unsetObjectStoreService(final ObjectStoreService objectStoreService) {
    if (_objectStoreService == objectStoreService) {
      _objectStoreService = null;
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addBucket(final BucketDefinition bucket) throws PersistenceException {
    if (_log.isInfoEnabled()) {
      _log.info("Adding bucket: " + bucket.getName());
    }
    try {
      final String objectId = BUCKETS_PREFIX + bucket.getName();
      final AnyMap bucketAny = bucket.toAny();
      writeObjectToStore(objectId, bucketAny);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error adding bucket", bucket.getName(), e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public BucketDefinition getBucket(final String name) throws PersistenceException {
    try {
      if (!existsObjectInStore(BUCKETS_PREFIX + name)) {
        return null;
      }
      final String objectId = BUCKETS_PREFIX + name;
      final AnyMap bucketAny = readObjectFromStore(objectId);
      return new BucketDefinition(bucketAny);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error getting bucket", name, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void removeBucket(final String name) throws PersistenceException {
    if (_log.isInfoEnabled()) {
      _log.info("Deleting bucket: " + name);
    }
    try {
      removeObjectFromStore(BUCKETS_PREFIX + name);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error deleting bucket", name, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getBuckets() throws PersistenceException {
    try {
      final Collection<String> bucketNames = readObjectNamesFromStore(BUCKETS_PREFIX);
      return bucketNames;
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error getting buckets", null, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addJob(final JobDefinition job) throws PersistenceException {
    if (_log.isInfoEnabled()) {
      _log.info("Adding job: " + job.getName());
    }
    try {
      final String objectId = JOBS_PREFIX + job.getName();
      final AnyMap jobAny = job.toAny();
      writeObjectToStore(objectId, jobAny);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error adding job", job.getName(), e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public JobDefinition getJob(final String name) throws PersistenceException {
    if (!hasJob(name)) {
      return null;
    }
    try {
      final String objectId = JOBS_PREFIX + name;
      final AnyMap jobAny = readObjectFromStore(objectId);
      return new JobDefinition(jobAny);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error getting job", name, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean hasJob(final String name) throws PersistenceException {
    try {
      return existsObjectInStore(JOBS_PREFIX + name);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error checking job", name, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void removeJob(final String name) throws PersistenceException {
    if (_log.isInfoEnabled()) {
      _log.info("Deleting job: " + name);
    }
    try {
      removeObjectFromStore(JOBS_PREFIX + name);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error deleting job", name, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getJobs() throws PersistenceException {
    try {
      final Collection<String> jobNames = readObjectNamesFromStore(JOBS_PREFIX);
      return jobNames;
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error getting jobs", null, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addWorkflow(final WorkflowDefinition workflow) throws PersistenceException {
    if (_log.isInfoEnabled()) {
      _log.info("Adding workflow: " + workflow.getName());
    }
    try {
      final String objectId = WORKFLOWS_PREFIX + workflow.getName();
      final AnyMap workflowAny = workflow.toAny();
      writeObjectToStore(objectId, workflowAny);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error adding workflow", workflow.getName(), e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public WorkflowDefinition getWorkflow(final String name) throws PersistenceException {
    try {
      if (!existsObjectInStore(WORKFLOWS_PREFIX + name)) {
        return null;
      }
      final String objectId = WORKFLOWS_PREFIX + name;
      final AnyMap workflowAny = readObjectFromStore(objectId);
      return new WorkflowDefinition(workflowAny);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error getting workflow", name, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getWorkflows() throws PersistenceException {
    try {
      final Collection<String> workflowNames = readObjectNamesFromStore(WORKFLOWS_PREFIX);
      return workflowNames;
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error getting workflows", null, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void removeWorkflow(final String name) throws PersistenceException {
    if (_log.isInfoEnabled()) {
      _log.info("Deleting workflow: " + name);
    }
    try {
      removeObjectFromStore(WORKFLOWS_PREFIX + name);
    } catch (final Exception e) {
      throw new PersistenceException(buildMessage("Error deleting workflow", name, e), e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getJobRunIds(final String jobName) throws PersistenceException {
    try {
      final String prefix = getJobRunDirectory(jobName);
      final Collection<String> jobRunIds = readObjectNamesFromStore(prefix);
      return jobRunIds;
    } catch (final Exception ex) {
      throw new PersistenceException(
        buildMessage("Error reading job run Ids from ObjectStore for job", jobName, ex), ex);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void storeJobRun(final String jobName, final String jobRunId, final AnyMap jobRunData) {
    final String objectId = getJobRunDataName(jobName, jobRunId);
    try {
      if (_log.isInfoEnabled()) {
        String jsonData;
        try {
          jsonData = _anyWriter.writeJsonObject(jobRunData);
        } catch (final IOException jsonex) {
          // fallback to toString, if for whatever reason we cannot produce real JSON.
          jsonData = jobRunData.toString();
        }
        _log.info("Job run data of run '" + jobRunId + "' for job '" + jobName + "': " + jsonData);
      }
      writeObjectToStore(objectId, jobRunData);
    } catch (final Exception ex) {
      _log.error("Failed to store job run data of run '" + jobRunId + "' for job '" + jobName + "'", ex);
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean containsJobRun(final String jobName, final String jobRunId) throws PersistenceException {
    final String path = getJobRunDataName(jobName, jobRunId);
    try {
      return existsObjectInStore(path);
    } catch (final Exception ex) {
      throw new PersistenceException(buildMessage("Error checking existence of run '" + jobRunId + "' for job",
        jobName, ex), ex);
    }
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getJobRunData(final String jobName, final String jobRunId) throws PersistenceException {
    final String objectId = getJobRunDataName(jobName, jobRunId);
    try {
      if (!existsObjectInStore(objectId)) {
        return null;
      }
      return readObjectFromStore(objectId);
    } catch (final Exception ex) {
      throw new PersistenceException(buildMessage("Error getting data for job run '" + jobRunId + "' of job",
        jobName, ex), ex);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void deleteJobRunData(final String jobName, final String jobRunId) throws PersistenceException {
    if (_log.isInfoEnabled()) {
      _log.info("Deleting job run data for job run: " + jobRunId);
    }
    final String path = getJobRunDataName(jobName, jobRunId);
    try {
      removeObjectFromStore(path);
    } catch (final Exception ex) {
      throw new PersistenceException(buildMessage("Error deleting data for job run '" + jobRunId + "' of job",
        jobName, ex), ex);
    }
  }

  // ----- private section -----

  /**
   * @param jobName
   *          the job
   * @return the objectstore directory where the job runs for the given job are stored
   */
  private String getJobRunDirectory(final String jobName) {
    return JOB_RUNS_PREFIX + jobName + '/';
  }

  /**
   * @param jobName
   *          the job
   * @param jobRunId
   *          the job
   * @return the objectstore directory where the job run data for the given job run is stored
   */
  private String getJobRunDataName(final String jobName, final String jobRunId) {
    return getJobRunDirectory(jobName) + jobRunId;
  }

  /**
   * Makes sure the store where the jobmanager data is stored, does exist and creates it with the store properties
   * retrieved from {@link ClusterConfigService} if it does not exist, yet.
   * 
   * @throws ObjectStoreException
   *           the store could not be created.
   * 
   */
  private void ensureStore() throws ObjectStoreException {
    if (!_isStorePrepared) {
      try {
        synchronized (this) {
          // yep, this is a double check which will be reported by FindBugs, but we opted to
          // first check if the store is prepared without synchronizing and then synchronizing
          // only if the store has not yet been prepared. But within that synchronized block we
          // wanted to avoid multiple store preparations by the then synchronized callers.
          if (!_isStorePrepared) {
            ObjectStoreRetryUtil.retryEnsureStore(_objectStoreService, JOBMANAGER_STORE);
            _isStorePrepared = true;
          }
        }
      } catch (final ServiceUnavailableException ex) {
        throw new ServiceUnavailableException(
          buildMessage("Finally failed to prepare store", JOBMANAGER_STORE, ex), ex);
      }
    }
  }

  /**
   * Removes an object from the JobMAnager store.
   * 
   * @param objectId
   *          the id of the object
   * @throws ObjectStoreException
   *           object could not be removed
   */
  private void removeObjectFromStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    _objectStoreService.removeObject(JOBMANAGER_STORE, objectId);
  }

  /**
   * Removes an object from the JobMAnager store.
   * 
   * @param objectId
   *          the id of the object
   * @throws ObjectStoreException
   *           object could not be removed
   */
  private boolean existsObjectInStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    return ObjectStoreRetryUtil.retryExistsObject(_objectStoreService, JOBMANAGER_STORE, objectId);
  }

  /**
   * @param objectId
   *          the object id under which the object is stored
   * @param object
   *          the object to store
   * @throws ObjectStoreException
   *           error writing object to the {@link ObjectStoreService}.
   */
  private void writeObjectToStore(final String objectId, final AnyMap object) throws ObjectStoreException {
    Exception retryableEx = null;
    ensureStore();
    for (int i = 0; i < MAX_RETRY_ON_STORE_UNAVAILABLE; i++) {
      StoreOutputStream storeOutputStream = null;
      try {
        storeOutputStream = _objectStoreService.writeObject(JOBMANAGER_STORE, objectId);
        _anyWriter.writeJsonStream(object, storeOutputStream);
        return;
      } catch (final IOException ex) {
        if (storeOutputStream != null) {
          storeOutputStream.abort();
        }
        retryableEx = ex;
        _log.warn("IOException on writing object '" + objectId + "', retrying: " + ex.toString());
      } catch (final ServiceUnavailableException ex) {
        retryableEx = ex;
        _log.warn("ServiceUnavailableException on writing object '" + objectId + "', retrying: " + ex.toString());
      } finally {
        IOUtils.closeQuietly(storeOutputStream);
      }
    }
    throw new ServiceUnavailableException(buildMessage("Finally failed to write object", objectId, retryableEx),
      retryableEx);
  }

  /**
   * @param objectId
   *          the object id of the object to read from store
   * @return the stored object for the given object id
   * @throws ObjectStoreException
   *           could not read object from the {@link ObjectStoreService}.
   */
  private AnyMap readObjectFromStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    Exception retryableEx = null;
    for (int i = 0; i < MAX_RETRY_ON_STORE_UNAVAILABLE; i++) {
      try {
        InputStream jobJsonStream = null;
        try {
          jobJsonStream = _objectStoreService.readObject(JOBMANAGER_STORE, objectId);
          return _recordReader.readJsonStream(jobJsonStream).getMetadata();
        } finally {
          IOUtils.closeQuietly(jobJsonStream);
        }
      } catch (final IOException ex) {
        retryableEx = ex;
        _log.warn("IOException on writing object '" + objectId + "', retrying: " + ex.toString());
      } catch (final ServiceUnavailableException ex) {
        retryableEx = ex;
        _log.warn("ServiceUnavailableException on writing object '" + objectId + "', retrying: " + ex.toString());
      }
    }
    throw new ServiceUnavailableException(buildMessage("Finally failed to read object", objectId, retryableEx),
      retryableEx);
  }

  /**
   * @param prefix
   *          the prefix for filtering the returned objects
   * @return the object names of the objects with the given prefix found in the store
   * @throws ObjectStoreException
   *           could not read object information from the {@link ObjectStoreService}.
   */
  private Collection<String> readObjectNamesFromStore(final String prefix) throws ObjectStoreException {
    ensureStore();
    final Collection<String> objectNames = new TreeSet<String>();
    for (final StoreObject info : _objectStoreService.getStoreObjectInfos(JOBMANAGER_STORE, prefix)) {
      final String objectName = info.getId().substring(prefix.length());
      objectNames.add(objectName);
    }
    return objectNames;
  }

  /** create a message text for PersistenceExceptions. */
  private String buildMessage(final String prefix, final String objectName, final Exception ex) {
    final StringBuilder message = new StringBuilder(prefix);
    if (objectName != null) {
      message.append(" '").append(objectName).append("'");
    }
    message.append(" due to ").append(ex.getClass().getSimpleName());
    message.append(": ").append(ex.getMessage());
    return message.toString();
  }

}
