/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/

package org.eclipse.smila.jobmanager;

import java.util.Collection;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.definitions.JobRunDefinitions;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;

/**
 * Provides job management and job run data.
 */
public interface JobRunDataProvider {

  /**
   * @param jobName
   *          name of the job for which to monitor the run.
   * @param jobId
   *          id of the job run to get the monitoring data. May be either the current run or a completed run.
   * @return monitoring data for the job run with the given id.
   * @throws JobManagerException
   *           error getting job run data
   */
  AnyMap getJobRunData(String jobName, String jobId) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job for which to monitor the run.
   * @param jobId
   *          id of the job run to get the monitoring data. May be either the current run or a completed run.
   * @param returnDetails
   *          whether to return details (e.g. workflow/bucket definition)
   * @return monitoring data for the job run with the given id.
   * @throws JobManagerException
   *           error getting job run data
   */
  AnyMap getJobRunData(String jobName, String jobId, boolean returnDetails) throws JobManagerException;

  /**
   * get information about a workflow run. The result contains these keys with integer values:
   * <ul>
   * <li>{@link org.eclipse.smila.jobmanager.definitions.JobManagerConstants#DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS}
   * <li>{@link org.eclipse.smila.jobmanager.definitions.JobManagerConstants#DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS}
   * </ul>
   * The result may contain additional keys.
   * 
   * @return workflow run information
   * 
   * @throws JobManagerException
   *           jobrun or workflowrun are not active anymore.
   */
  AnyMap getWorkflowRunData(String jobName, String jobId, String workflowRunId) throws JobManagerException;

  /**
   * get info on current job run.
   * 
   * @param jobName
   *          job name
   * @return if job is defined and active: id and state of current run. Else null.
   * @throws JobManagerException
   *           error getting state.
   */
  JobRunInfo getJobRunInfo(String jobName) throws JobManagerException;

  /**
   * get Ids of complete job runs.
   * 
   * @param jobName
   *          job name
   * @return Ids of completed job runs, sorted lexicographically ascending. Empty set if job is not defined or no
   *         completed job runs exists.
   * @throws JobManagerException
   *           error getting state.
   */
  Collection<String> getCompletedJobRunIds(String jobName) throws JobManagerException;

  /**
   * @return the currently persisted job/workflow/bucket definitions for the given job.
   */
  JobRunDefinitions getJobRunDefinitions(final String jobName) throws JobManagerException;

  /**
   * Set custom data for given job (run), worker and path. If not needed, 'path' parameter can be left out. If given
   * path already exists for given job (run) and worker, existing data will be overwritten.
   * 
   * @throws JobManagerException
   *           if job (run) does not exist or state of job run does not allow setting custom data.
   */
  void setCustomData(String jobName, String jobRunId, String worker, byte[] data, String... path)
    throws JobManagerException;

  /**
   * @return custom data for given job (run), worker and path. Returns 'null' if no data is stored or job (run), worker
   *         or path do not exist.
   */
  byte[] getCustomData(String jobName, String jobRunId, String worker, String... path) throws JobManagerException;;

}
