/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.definitions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.common.definitions.ParameterDefinition;
import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.taskworker.input.InputMode;
import org.eclipse.smila.taskworker.output.OutputMode;

/**
 * Definition for workers e.g.
 * 
 * <pre>
 * { "workers":
 *   [
 *     {  "name": "worker",
 *        "modes: ["bulkSource", "autoCommit"],
 *        "parameters": [ {"name": "parameter"} ],
 *        "output":
 *          [
 *            {  "name": "insertedRecords",
 *               "type": "recordBulks",
 *               "modes": [ "optional" ],
 *               "group": "recordBulk"
 *            }
 *          ]
 *      }, ...
 *   ]
 * }
 * </pre>
 */
public class WorkerDefinition extends DefinitionBase {

  /**
   * Worker modes.
   */
  public enum Mode {
    /**
     * Can start a workflow, does not need input data. A task for this worker is created on demand when the worker
     * requests it.
     */
    BULKSOURCE("bulkSource"),

    /**
     * When the worker dies while working on a task (sends no keep-alive anymore) the started bulks are committed by the
     * job manager and follow-up actions are triggered, the task is not rolled back.
     */
    AUTOCOMMIT("autoCommit"),
    /**
     * Task delivery to this worker should not be limited by scale-up control.
     */
    RUNALWAYS("runAlways"),
    /**
     * Worker wants to take part in a completion run.
     */
    REQUESTSCOMPLETION("requestsCompletion"),
    /**
     * All tasks in the current workflow run for workers before this one have to be finished before a task for this one
     * is generated.
     */
    BARRIER("barrier");

    /** the name of the Mode. */
    private String _name;

    /**
     * constructs a Mode.
     * 
     * @param name
     *          The name of the mode.
     */
    Mode(final String name) {
      _name = name;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      return _name;
    }
  };

  /**
   * A helper class to represent Input or Output nodes for worker definitions.
   * 
   * @param <T>
   *          The Modes to use.
   */
  public abstract static class InputOutput<T extends Enum<T>> {
    /** The name of the input (mandatory). */
    private final String _name;

    /** The data object type of the input or output (mandatory). */
    private final String _type;

    /** The modes of the input (optional). */
    private final Collection<T> _modes = new ArrayList<T>();

    /**
     * Constructs an input or putput from an Any.
     * 
     * @param any
     *          The Any object from which this instance should be constructed.
     * @throws InvalidDefinitionException
     *           An error occurred during the conversion of the Any object.
     */
    public InputOutput(final AnyMap any) throws InvalidDefinitionException {
      _name = AccessAny.getStringRequired(any, KEY_NAME);
      _type = AccessAny.getStringRequired(any, KEY_TYPE);
      // modes
      final List<String> modeNames = AccessAny.getStringSeq(any, KEY_MODES);
      if (modeNames != null) {
        for (final String modeName : modeNames) {
          try {
            _modes.add(getMode(modeName.toUpperCase(Locale.US)));
          } catch (final IllegalArgumentException ex) {
            throw new InvalidDefinitionException("Invalid worker definition mode name " + modeName + " for worker "
              + _name);
          }
        }
      }
    }

    /**
     * Gets the Mode for the given modeName.
     * 
     * @param modeName
     *          The mode name.
     * @return The Mode (instance of T)
     */
    protected abstract T getMode(String modeName);

    /**
     * @return the name
     */
    public String getName() {
      return _name;
    }

    /**
     * @return the type
     */
    public String getType() {
      return _type;
    }

    /**
     * @return the modes
     */
    public Collection<T> getModes() {
      return AccessAny.copyCollection(_modes);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      return String.format("name: \"%s\", type: \"%s\", modes %s", _name, _type, _modes);
    }

    /**
     * Converts the object to an Any.
     * 
     * @return The object as Any.
     */
    public Any toAny() {
      final AnyMap resultAny = AccessAny.FACTORY.createAnyMap();
      resultAny.put(KEY_NAME, getName());
      resultAny.put(KEY_TYPE, getType());
      if (!getModes().isEmpty()) {
        final AnySeq modes = AccessAny.FACTORY.createAnySeq();
        for (final T mode : getModes()) {
          modes.add(mode.toString());
        }
        resultAny.put(KEY_MODES, modes);
      }
      return resultAny;
    }
  }

  /**
   * represents an Input node.
   */
  public static class Input extends InputOutput<InputMode> {

    /**
     * Constructs an Input.
     * 
     * @param inputAny
     *          The Any object from which this input should be constructed.
     * @throws InvalidDefinitionException
     *           An error occurred during the conversion of the Any input.
     */
    public Input(final AnyMap inputAny) throws InvalidDefinitionException {
      super(inputAny);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected InputMode getMode(final String modeName) {
      return InputMode.valueOf(modeName);
    }
  }

  /**
   * represents an Output node.
   */
  public static class Output extends InputOutput<OutputMode> {

    /** The Group of the input or output (optional) slot. */
    private final String _group;

    /**
     * Constructs an Ouptut.
     * 
     * @param outputAny
     *          The Any object from which this output should be constructed.
     * @throws InvalidDefinitionException
     *           An error occurred during the conversion of the Any object.
     */
    public Output(final AnyMap outputAny) throws InvalidDefinitionException {
      super(outputAny);
      // slot group (if any)
      _group = AccessAny.getString(outputAny, KEY_GROUP);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected OutputMode getMode(final String modeName) {
      return OutputMode.valueOf(modeName);
    }

    /**
     * @return the _group
     */
    public String getGroup() {
      return _group;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Any toAny() {
      final AnyMap resultAny = (AnyMap) super.toAny();
      if (_group != null) {
        resultAny.put(KEY_GROUP, _group);
      }
      return resultAny;
    }
  }

  /** Property for workers in json. */
  public static final String KEY_WORKERS = "workers";

  /** Property for modes in json. */
  public static final String KEY_MODES = "modes";

  /** Property for parameters in json. */
  public static final String KEY_PARAMETERS = "parameters";

  /** Property for output in json. */
  public static final String KEY_OUTPUT = "output";

  /** Property for input in json. */
  public static final String KEY_INPUT = "input";

  /** Property for type in json. */
  public static final String KEY_TYPE = "type";

  /** Property for taskGenerator in json. */
  public static final String KEY_TASK_GENERATOR = "taskGenerator";

  /** Property for input or output slot group in json. */
  public static final String KEY_GROUP = "group";

  /** The modes of the worker definition (optional). */
  private final Collection<Mode> _modes = new ArrayList<Mode>();

  /** The parameters of the worker definition (optional). */
  private final List<ParameterDefinition> _parameters;

  /** The inputs of the worker definition (optional). */
  private final Collection<Input> _input = new ArrayList<Input>();

  /** The outputs of the worker definition (optional). */
  private final Collection<Output> _output = new ArrayList<Output>();

  /** The task generator of the worker definition (optional). */
  private final String _taskGenerator;

  /**
   * Constructor for WorkerDefinition.
   * 
   * @param workerAny
   *          The any Object with one WorkerDefinition
   * @throws InvalidDefinitionException
   *           exception if the any object is not filled with all desired values
   */
  public WorkerDefinition(final AnyMap workerAny) throws InvalidDefinitionException {
    super(workerAny);
    checkParameters(_definitionMap.get(KEY_PARAMETERS));
    if (_definitionMap.containsKey(KEY_PARAMETERS)) {
      _parameters = ParameterDefinition.parseParameters(_definitionMap.getSeq(KEY_PARAMETERS));
    } else {
      _parameters = new ArrayList<ParameterDefinition>();
    }
    _taskGenerator = _definitionMap.getStringValue(KEY_TASK_GENERATOR);
    // modes
    final List<String> modeNames = AccessAny.getStringSeq(_definitionMap, KEY_MODES);
    if (modeNames != null) {
      for (final String modeName : modeNames) {
        try {
          _modes.add(Mode.valueOf(modeName.toUpperCase(Locale.US)));
        } catch (final IllegalArgumentException ex) {
          throw new InvalidDefinitionException("Invalid worker definition mode name " + modeName + " for worker "
            + _name);
        }
      }
    }
    parseInputs();
    parseOutputs();
  }

  private void parseInputs() throws InvalidDefinitionException {
    final AnySeq inputList = (AnySeq) AccessAny.get(_definitionMap, KEY_INPUT, ValueType.SEQ);
    if (inputList != null) {
      try {
        for (final Any inputAny : inputList) {
          _input.add(new Input((AnyMap) inputAny));
        }
      } catch (final Exception e) {
        throw new InvalidDefinitionException("Invalid input for worker description " + _name, e);
      }
    }
  }

  private void parseOutputs() throws InvalidDefinitionException {
    final AnySeq outputList = (AnySeq) AccessAny.get(_definitionMap, KEY_OUTPUT, ValueType.SEQ);
    if (outputList != null) {
      try {
        for (final Any outputAny : outputList) {
          _output.add(new Output((AnyMap) outputAny));
        }
      } catch (final Exception e) {
        throw new InvalidDefinitionException("Invalid output for worker description " + _name, e);
      }
    }
  }

  /**
   * @return the modes
   */
  public Collection<Mode> getModes() {
    return AccessAny.copyCollection(_modes);
  }

  /**
   * @return the names of the mandatory parameters
   */
  public Collection<String> getMandatoryParameterNames() {
    final Collection<String> mandatoryParameters = new ArrayList<>();
    if (_parameters != null) {
      for (final ParameterDefinition parameter : _parameters) {
        if (!parameter.isOptional()) {
          mandatoryParameters.add(parameter.getName());
        }
      }
    }
    return mandatoryParameters;
  }

  /**
   * @return the parameters
   */
  public List<ParameterDefinition> getParameters() {
    return AccessAny.copyCollection(_parameters);
  }

  /**
   * @return the parameters with given range
   */
  public List<ParameterDefinition> getParametersByRange(final String range) {
    final List<ParameterDefinition> result = new ArrayList<>();
    for (final ParameterDefinition parameterDef : _parameters) {
      final String paramRange = parameterDef.getRange();
      if ((range == null || range.isEmpty()) && (paramRange == null || paramRange.isEmpty()) || range != null
        && range.equals(paramRange)) {
        result.add(parameterDef);
      }
    }
    return result;
  }

  /**
   * @return the task generator
   */
  public String getTaskGenerator() {
    return _taskGenerator;
  }

  /**
   * @return the input
   */
  public Collection<Input> getInput() {
    return AccessAny.copyCollection(_input);
  }

  /**
   * @return the output
   */
  public Collection<Output> getOutput() {
    return AccessAny.copyCollection(_output);
  }

  /**
   * @return the output
   */
  public Map<String, Collection<OutputMode>> getOutputModes() {
    final Map<String, Collection<OutputMode>> result = new HashMap<>();
    if (_output != null) {
      for (final Output out : _output) {
        result.put(out.getName(), out.getModes());
      }
    }
    return result;
  }

  /**
   * Converts the object to an Any.
   * 
   * @return The object as Any.
   */
  @Override
  public AnyMap toAny(final boolean includingAdditionalAttributes) {
    try {
      final AnyMap defAny = super.toAny(includingAdditionalAttributes);
      if (!includingAdditionalAttributes) {
        if (_parameters != null && !_parameters.isEmpty()) {
          final AnySeq paramSeq = ParameterDefinition.toAny(_parameters, includingAdditionalAttributes);
          defAny.put(KEY_PARAMETERS, paramSeq);
        }
        if (getTaskGenerator() != null) {
          defAny.put(KEY_TASK_GENERATOR, getTaskGenerator());
        }
        if (!getModes().isEmpty()) {
          final Collection<String> modes = new ArrayList<String>();
          for (final Mode mode : getModes()) {
            modes.add(mode.toString());
          }
          defAny.put(KEY_MODES, AnyUtil.objectToAny(modes));
        }

        addIOsToAnyMap(defAny, KEY_INPUT, getInput());
        addIOsToAnyMap(defAny, KEY_OUTPUT, getOutput());
      }
      return defAny;
    } catch (final Exception ex) {
      throw new IllegalStateException("Failed to create Any object for data object type mode", ex);
    }
  }

  private void addIOsToAnyMap(final AnyMap defAny, final String key, final Collection<? extends InputOutput<?>> ios) {
    if (!ios.isEmpty()) {
      final AnySeq ioAny = defAny.getSeq(key, true);
      for (final InputOutput<?> io : ios) {
        ioAny.add(io.toAny());
      }
    }
  }

  /**
   * @param slotName
   *          The slot name
   * @return The input with the given slot name
   */
  public Input getInput(final String slotName) {
    if (_input != null) {
      for (final Input iterableElement : _input) {
        if (iterableElement.getName().equals(slotName)) {
          return iterableElement;
        }
      }
    }
    return null;
  }

  /**
   * @param slotName
   *          The slot name
   * @return The output with given slot name
   */
  public Output getOutput(final String slotName) {
    if (_output != null) {
      for (final Output iterableElement : _output) {
        if (iterableElement.getName().equals(slotName)) {
          return iterableElement;
        }
      }
    }
    return null;
  }

  /**
   * @return true iff each group has mandatory slots. false if there are not groups or at least one group with only
   *         optional slots.
   */
  public boolean hasOnlyMandatoryGroups() {
    final Set<String> mandatoryGroups = new HashSet<String>();
    final Set<String> optionalGroups = new HashSet<String>();
    for (final Output output : _output) {
      final String group = output.getGroup();
      if (group != null) {
        if (output.getModes().contains(OutputMode.OPTIONAL)) {
          optionalGroups.add(group);
        } else {
          mandatoryGroups.add(group);
        }
      }
    }
    optionalGroups.removeAll(mandatoryGroups);
    return !mandatoryGroups.isEmpty() && optionalGroups.isEmpty();
  }

  /**
   * checks the parameters for validity.
   * 
   * @param parameters
   *          the parameters to check
   * @throws InvalidDefinitionException
   *           the parameters do not apply to the vaildation rules
   */
  private void checkParameters(final Any parameters) throws InvalidDefinitionException {
    if (parameters != null) {
      // check if sequence.
      if (!parameters.isSeq()) {
        throw new InvalidDefinitionException("invalid parameters format. Only Sequence of Maps allowed.");
      }
      final AnySeq parameterSeq = parameters.asSeq();
      for (final Any singleParam : parameterSeq) {
        if (singleParam.isMap()) {
          final Any nameOfParam = singleParam.asMap().get(KEY_NAME);
          if (nameOfParam == null || !nameOfParam.isString()) {
            throw new InvalidDefinitionException("Parameter found without '" + KEY_NAME
              + "' argument of type string: " + singleParam);
          }
        } else {
          throw new InvalidDefinitionException("invalid parameters format. Only Sequence of Maps allowed.");
        }
      }
    }
  }
}
