/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.output;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.io.IODataObject;
import org.eclipse.smila.taskworker.io.IODataObjects;
import org.eclipse.smila.taskworker.util.Counters;

/**
 * Returned by {@link org.eclipse.smila.taskworker.TaskContext#getOutputs()} this class provides access to the data
 * objects associated by a task to the output slots of a worker. It can create different kinds of {@link Output}
 * wrappers on these data objects that make access for the workers easier. You can create only a single output wrapper
 * for each data object. On the second call, only null will be returned.
 * 
 * 
 * @see StreamOutput
 * @see RecordOutput
 * @see KvoOutput
 * @see AppendableOutput
 */

public class Outputs extends IODataObjects {
  /** duration for committing all open output objects. */
  private long _durationCommit;

  /** key: output slot name, value: output modes. */
  private Map<String, Collection<OutputMode>> _slotOutputModes = new HashMap<String, Collection<OutputMode>>(0);

  /**
   * @param dataObjects
   *          The data objects
   * @param objectStore
   *          The reference to the object store service
   */
  public Outputs(final Map<String, List<BulkInfo>> dataObjects, final ObjectStoreService objectStore) {
    super(dataObjects, objectStore);
  }

  /**
   * @param outputModes
   *          the output modes for the output slots.
   */
  public void setOutputModes(final Map<String, Collection<OutputMode>> outputModes) {
    _slotOutputModes = outputModes;
  }

  /**
   * Generic method to create output wrapper for the first data object in a slot. See
   * {@link #getAsOutput(String, int, Class)} for details, this method just calls this method with
   * <code>index = 0</code>
   * 
   * @return first data object in given slot prepared for custom access. Null, if slot is not set, or an output wrapper
   *         has already been created earlier.
   * @throws IllegalArgumentException
   *           if the outputClass cannot be instantiated.
   */
  public <T extends Output> T getAsOutput(final String slotName, final Class<T> inputClass) {
    return getAsOutput(slotName, 0, inputClass);
  }

  /**
   * Generic method to create output wrappers. Any predefined subclass of {@link Output} can be used as well as custom
   * defined class if you need it (be sure to implement the necessary constructor). For example, to get an
   * Key-Value-Object wrapper for the first object in slot "kvoOutputSlot", use
   * <code>KvoOutput kvo = taskContext.getOutputs().getAsOutput("kvoOutputSlot", KvoOutput.class);</code>
   * 
   * @see StreamOutput
   * @see RecordOutput
   * @see KvoOutput
   * @see AppendableOutput
   * 
   * @return n-th data object in given slot prepared for custom access. Null, if slot is not set, has less objects, or a
   *         wrapper has already been created earlier.
   * @throws IllegalArgumentException
   *           if the outputClass cannot be instantiated.
   */
  public <T extends Output> T getAsOutput(final String slotName, final int index, final Class<T> outputClass) {
    if (canCreate(slotName, index)) {
      final BulkInfo dataObject = getDataObject(slotName, index);
      try {
        final Constructor<T> outputConstructor =
          outputClass.getConstructor(BulkInfo.class, ObjectStoreService.class);
        final T output = outputConstructor.newInstance(dataObject, getObjectStore());
        putIOData(slotName, index, output);
        return output;
      } catch (final Exception ex) {
        throw new IllegalArgumentException("Canot create instance of class " + outputClass, ex);
      }
    }
    return null;
  }

  /**
   * @param slotName
   *          The slot name
   * @param index
   *          The index
   * @return The data object as {@link StreamOutput} with given slot name and index
   */
  public StreamOutput getAsStreamOutput(final String slotName, final int index) {
    return getAsOutput(slotName, index, StreamOutput.class);
  }

  /**
   * @param slotName
   *          The slot name
   * @return The first data object as {@link StreamOutput} with given slot name
   */
  public StreamOutput getAsStreamOutput(final String slotName) {
    return getAsStreamOutput(slotName, 0);
  }

  /**
   * @param slotName
   *          The slot name
   * @param index
   *          The index
   * @return The data object as {@link RecordOutput} with given slot name and index
   */
  public RecordOutput getAsRecordOutput(final String slotName, final int index) {
    return getAsOutput(slotName, index, RecordOutput.class);

  }

  /**
   * @param slotName
   *          The slot name
   * @return The first data object as {@link RecordOutput} with given slot name
   */
  public RecordOutput getAsRecordOutput(final String slotName) {
    return getAsRecordOutput(slotName, 0);
  }

  /**
   * Commit all open data objects, if necessary. Called by WorkerManager after task has finished.
   */
  public void commitAll() throws ObjectStoreException, IOException {
    for (final List<IODataObject> outputs : getIOData().values()) {
      for (final IODataObject output : outputs) {
        if (output != null) {
          final long startTime = System.nanoTime();
          ((Output) output).commit();
          _durationCommit += (System.nanoTime() - startTime);
        }
      }
    }
  }

  /**
   * Abort all open data objects, if necessary. Called by WorkerManager after task has finished.
   */
  public void abortAll() throws ObjectStoreException, IOException {
    for (final List<IODataObject> outputs : getIOData().values()) {
      for (final IODataObject output : outputs) {
        if (output != null) {
          final long startTime = System.nanoTime();
          ((Output) output).abort();
          _durationCommit += (System.nanoTime() - startTime);
        }
      }
    }
  }

  /** aggregate counters of output slots. Called by WorkerManager after task has finished. */
  public void addOutputCounters(final Map<String, Number> counters) {
    if (_durationCommit > 0) {
      Counters.addDuration(counters, Counters.DURATION_IODATA_CLOSE, _durationCommit);
    }
    addCounters(counters, Counters.OUTPUT);
  }

  /** @return whether given slot can have multiple output objects. */
  protected boolean isMultiple(final String slotName) {
    if (_slotOutputModes.containsKey(slotName) && _slotOutputModes.get(slotName).contains(OutputMode.MULTIPLE)) {
      return true;
    }
    return false;
  }

  @Override
  protected boolean canCreate(final String slotName, final int index) {
    if (isMultiple(slotName)) {
      List<IODataObject> slot = _ioData.get(slotName);
      if (slot == null) {
        slot = new ArrayList<IODataObject>();
        _ioData.put(slotName, slot);
      }
      return slot.size() <= index;
    }
    return super.canCreate(slotName, index);
  }

  @Override
  public BulkInfo getDataObject(final String slotName, final int index) {
    if (isMultiple(slotName)) {
      final BulkInfo template = _dataObjects.get(slotName).get(0);
      final String objectNamePrefix = template.getObjectName();
      BulkInfo result = null;
      if (index == 0) {
        // the first object should also have a suffix, we don't want to use the prefix as object name
        result = new BulkInfo(template.getBucketName(), template.getStoreName(), objectNamePrefix + index);
      } else if (_dataObjects.get(slotName).size() <= index) {
        // the given object name is just used as a prefix for the created object.
        result = new BulkInfo(template.getBucketName(), template.getStoreName(), objectNamePrefix + index);
        _dataObjects.get(slotName).add(index, result);
      } else {
        result = _dataObjects.get(slotName).get(index);
      }
      return result;
    }
    return super.getDataObject(slotName, index);
  }
}
