/**********************************************************************************************
 * Copyright (c) 2008,2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * <ul>
 * <li>Juergen Schumacher (empolis GmbH) - initial API and implementation</li>
 * <li>Andreas Weber (Attensity Europe GmbH) renamed LiteralFormatHelper -> ValueFormatHelper</li>
 * <li>Marco Strack (Empolis Information Management GmbH) - included ISO8601 time zone formats</li>
 * </ul>
 **********************************************************************************************/

package org.eclipse.smila.datamodel;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.regex.Pattern;

/**
 * helper class for formatting and parsing Values. all methods synchronize on the used local formatter object, so you
 * can use the shared instance. Using multiple instances may improve performance, though, because of less
 * synchronization.
 * 
 * See <a href="http://wiki.eclipse.org/SMILA/Documentation/Data_Model_and_Serialization_Formats">SMILA Data Model</a>
 * for details.
 * 
 * 
 */
public class ValueFormatHelper {
  /** shared global helper instance. */
  public static final ValueFormatHelper INSTANCE = new ValueFormatHelper();

  /** valid pattern for strings to be parsed as date. */
  private static final String DATE_PATTERN = "yyyy-MM-dd";

  /** The length of strings to be parsed as date. */
  private static final int DATE_LENGTH = DATE_PATTERN.length();

  /** The length of strings to be parsed as date time for (default) pattern (TZ denoted as +0330). */
  private static final int DATE_TIME_LENGTH_PATTERN_DEFAULT = 28;

  /** The length of strings to be parsed as date time short plus Z. */
  private static final int DATE_TIME_LENGTH_PATTERN_SHORT_Z = 20;

  /** The length of strings to be parsed as date time long plus Z. */
  private static final int DATE_TIME_LENGTH_PATTERN_LONG_Z = 24;

  /** The length of strings to be parsed as date time for pattern 2. */
  private static final int DATE_TIME_LENGTH_PATTERN_2 = 22;

  /** The length of strings to be parsed as date time for pattern 3. */
  private static final int DATE_TIME_LENGTH_PATTERN_3 = 24;

  /** The length of strings to be parsed as date time for pattern 4. */
  private static final int DATE_TIME_LENGTH_PATTERN_4 = 25;

  /** The length of strings to be parsed as date time for pattern 5. */
  private static final int DATE_TIME_LENGTH_PATTERN_5 = 26;

  /** The length of strings to be parsed as date time for pattern 6. */
  private static final int DATE_TIME_LENGTH_PATTERN_6 = 28;

  /** The length of strings to be parsed as date time for pattern 7. */
  private static final int DATE_TIME_LENGTH_PATTERN_7 = 29;

  /** formatter to create and parse standard string representations of Date values: "yyyy-MM-dd". */
  private final DateFormat _formatDate = new SimpleDateFormat(DATE_PATTERN);

  /** valid Datetime value pattern with milliseconds and time zone (default for printing). */
  private final DateFormat _formatDateTimePatternDefault = getDefaultDateTimeFormat();

  /** baselen w/o TZ is 19 */
  /** valid Datetime value pattern with time zone. (+10 / Z) (len 20 or 22) */
  private final DateFormat _formatDateTimePattern2 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssX");

  /** valid Datetime value pattern with time zone. (+1000 / Z) (len 20 or 24) */
  private final DateFormat _formatDateTimePattern3 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssXX");

  /** valid Datetime value pattern with time zone. (+10:00 / Z) (len 20 or 25) */
  private final DateFormat _formatDateTimePattern4 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssXXX");

  /** baselen with ms and w/o TZ is 23 */
  /** valid Datetime value pattern with milliseconds and time zone. (+10 / Z) (len 24 or 26) */
  private final DateFormat _formatDateTimePattern5 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSX");

  /** valid Datetime value pattern with milliseconds and time zone. (+1000 / Z) (len 24 or 28) */
  private final DateFormat _formatDateTimePattern6 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXX");

  /** valid Datetime value pattern with milliseconds and time zone. (+10:00 / Z) (len 24 or 29) */
  private final DateFormat _formatDateTimePattern7 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");

  /** pattern for checking if the string might be a date. */
  private final Pattern _formatPatternDateDefault = Pattern.compile("\\d{4}-\\d{2}-\\d{2}");

  /** pattern for checking if the string might be a date time for default pattern. */
  private final Pattern _formatPatternDateTimeDefault = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}\\.\\d{3}[+-]{1}\\d{4}");

  /** pattern for checking if the string might be be a UTC date time w/o ms. */
  private final Pattern _formatPatternDateTimeShortZ = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}Z");

  /** pattern for checking if the string might be a UTC date time inclu. ms. */
  private final Pattern _formatPatternDateTimeLongZ = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}\\.\\d{3}Z");

  /** pattern for checking if the string might be a date time for pattern 2. */
  private final Pattern _formatPatternDateTime2 = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}[+-]\\d{2}");

  /** pattern for checking if the string might be a date time for pattern 3. */
  private final Pattern _formatPatternDateTime3 = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}[+-]\\d{4}");

  /** pattern for checking if the string might be a date time for pattern 4. */
  private final Pattern _formatPatternDateTime4 = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}[+-]\\d{2}:\\d{2}");

  /** pattern for checking if the string might be a date time for pattern 5. */
  private final Pattern _formatPatternDateTime5 = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}\\.\\d{3}[+-]\\d{2}");

  /** pattern for checking if the string might be a date time for pattern 6. */
  private final Pattern _formatPatternDateTime6 = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}\\.\\d{3}[+-]\\d{4}");

  /** pattern for checking if the string might be a date time for pattern 7. */
  private final Pattern _formatPatternDateTime7 = Pattern
    .compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}\\.\\d{3}[+-]\\d{2}:\\d{2}");

  /**
   * create local instance.
   */
  public ValueFormatHelper() {
    _formatDate.setLenient(false);
    _formatDateTimePatternDefault.setLenient(false);
    final TimeZone utcTimeZone = TimeZone.getTimeZone("UTC");
    _formatDateTimePattern2.setLenient(false);
    _formatDateTimePattern2.setTimeZone(utcTimeZone);
    _formatDateTimePattern3.setLenient(false);
    _formatDateTimePattern3.setTimeZone(utcTimeZone);
    _formatDateTimePattern4.setLenient(false);
    _formatDateTimePattern4.setTimeZone(utcTimeZone);
    _formatDateTimePattern5.setLenient(false);
    _formatDateTimePattern5.setTimeZone(utcTimeZone);
    _formatDateTimePattern6.setLenient(false);
    _formatDateTimePattern6.setTimeZone(utcTimeZone);
    _formatDateTimePattern7.setLenient(false);
    _formatDateTimePattern7.setTimeZone(utcTimeZone);
  }

  /**
   * @return the default format for datetime values.
   */
  public static SimpleDateFormat getDefaultDateTimeFormat() {
    final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
    sdf.setLenient(false);
    return sdf;
  }

  /**
   * format value as Date string.
   * 
   * @param value
   *          a date value
   * @return formatted date.
   */
  public String formatDate(final Date value) {
    synchronized (_formatDate) {
      return _formatDate.format(value);
    }
  }

  /**
   * format value as DateTime string.
   * 
   * @param value
   *          a datetime value
   * @return formatted datetime string
   */
  public String formatDateTime(final Date value) {
    synchronized (_formatDateTimePatternDefault) {
      return _formatDateTimePatternDefault.format(value);
    }
  }

  /**
   * parse a date string.
   * 
   * @param dateString
   *          a date string
   * @return parsed Date
   * @throws ParseException
   *           string has wrong format
   */
  public Date parseDate(final String dateString) throws ParseException {
    if (dateString.length() == DATE_LENGTH && _formatPatternDateDefault.matcher(dateString).matches()) {
      synchronized (_formatDate) {
        return _formatDate.parse(dateString);
      }
    } else {
      throw new ParseException("Pattern of string '" + dateString + "' doesn't match supported date pattern "
        + DATE_PATTERN, 0);
    }
  }

  /**
   * parse a date string.
   * 
   * @param dateString
   *          a date string
   * @return parsed Date if correct format, else null.
   */
  public Date tryParseDate(final String dateString) {
    if (dateString.length() == DATE_LENGTH && _formatPatternDateDefault.matcher(dateString).matches()) {
      synchronized (_formatDate) {
        try {
          return _formatDate.parse(dateString);
        } catch (final ParseException ex) {
          ; // not a date ... fine.
        }

      }
    }
    return null;
  }

  /**
   * try to parse date or date time string (internal). Does not do any control flow.
   * 
   * @param dateTimeString
   *          a datetime string
   * @return parsed Date, null if invalid date
   * @throws ParseException
   *           string has wrong format
   */
  public Date parseDateTimeInternal(final String dateTimeString) throws ParseException {
    Date result = null;
    final int dateLen = dateTimeString.length();
    if (dateLen == DATE_TIME_LENGTH_PATTERN_DEFAULT
      && _formatPatternDateTimeDefault.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePatternDefault) {
        result = _formatDateTimePatternDefault.parse(dateTimeString);
      }
    } else if (dateLen == DATE_TIME_LENGTH_PATTERN_SHORT_Z
      && _formatPatternDateTimeShortZ.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePattern2) {
        result = _formatDateTimePattern2.parse(dateTimeString);
      }
    } else if (dateLen == DATE_TIME_LENGTH_PATTERN_LONG_Z
      && _formatPatternDateTimeLongZ.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePattern5) {
        result = _formatDateTimePattern5.parse(dateTimeString);
      }
    } else if (dateLen == DATE_TIME_LENGTH_PATTERN_2 /* short */
      && _formatPatternDateTime2.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePattern2) {
        result = _formatDateTimePattern2.parse(dateTimeString);
      }
    } else if (dateLen == DATE_TIME_LENGTH_PATTERN_3 && _formatPatternDateTime3.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePattern3) {
        result = _formatDateTimePattern3.parse(dateTimeString);
      }
    } else if (dateLen == DATE_TIME_LENGTH_PATTERN_4 && _formatPatternDateTime4.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePattern4) {
        result = _formatDateTimePattern4.parse(dateTimeString);
      }
    } else if (dateLen == DATE_TIME_LENGTH_PATTERN_5 /* long */
      && _formatPatternDateTime5.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePattern5) {
        result = _formatDateTimePattern5.parse(dateTimeString);
      }
    } else if (dateLen == DATE_TIME_LENGTH_PATTERN_6 && _formatPatternDateTime6.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePattern6) {
        result = _formatDateTimePattern6.parse(dateTimeString);
      }
    } else if (dateLen == DATE_TIME_LENGTH_PATTERN_7 && _formatPatternDateTime7.matcher(dateTimeString).matches()) {
      synchronized (_formatDateTimePattern7) {
        result = _formatDateTimePattern7.parse(dateTimeString);
      }
    }
    return result;
  }

  /**
   * parse date or date time string.
   * 
   * @param dateTimeString
   *          a datetime string
   * @return parsed Date
   * @throws ParseException
   *           string has wrong format
   */
  public Date parseDateTime(final String dateTimeString) throws ParseException {
    final Date result = parseDateTimeInternal(dateTimeString);
    if (result == null) {
      throw new ParseException("Pattern of datetime string '" + dateTimeString
        + "' doesn't match supported patterns", 0);
    }
    return result;
  }

  /**
   * try to parse date or date time string.
   * 
   * @param dateTimeString
   *          a datetime string
   * @return parsed Date if correct format, else null;
   */
  public Date tryParseDateTime(final String dateTimeString) {
    Date result = null;
    final int strLen = dateTimeString.length();

    /* rudimentary quick filters */
    if (strLen < DATE_LENGTH || strLen > DATE_TIME_LENGTH_PATTERN_7) {
      return null;
    }

    final int offsetT = 10;

    if (strLen > DATE_TIME_LENGTH_PATTERN_SHORT_Z) {
      if (!dateTimeString.substring(offsetT, offsetT + 1).equals("T")) {
        return null;
      }
    }

    try {
      result = parseDateTimeInternal(dateTimeString);
    } catch (final ParseException ex) {
      ; // not a datetime ... fine.
    }
    return result;
  }
}
