/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnySeq;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;

/**
 * Wraps an AnySeq as a Javascript object. Apart from accessing elements using numerical indexes, it supports the JS
 * Array properties "length" and "constructor". Adding elements is supported by putting an object to index
 * "wrapper.length".
 */
public class AnySeqWrapper extends ObjectWrapperBase {

  private static final String PROP_LENGTH = "length";

  private static final String PROP_CONSTRUCTOR = "constructor";

  private final AnySeq _seq;

  /** Creates instance. */
  public AnySeqWrapper(final AnySeq seq, final Scriptable parentScope) {
    super(parentScope);
    setPrototype(ScriptableObject.getArrayPrototype(parentScope));
    _seq = seq;
  }

  @Override
  public AnySeq unwrap() {
    return _seq;
  }

  @Override
  public String getClassName() {
    return "AnySeqWrapper";
  }

  @Override
  public Object get(final String name, final Scriptable start) {
    switch (name) {
      case PROP_LENGTH:
        return _seq.size();
      case PROP_CONSTRUCTOR: // called by debugger
        return null;
      default:
        throw Context.reportRuntimeError("Getting string keys is not allowed for Sequences: " + name);
    }
  }

  @Override
  public Object get(final int index, final Scriptable start) {
    return AnyWrapper.wrapAny(_seq.get(index), getParentScope());
  }

  @Override
  public boolean has(final String name, final Scriptable start) {
    return PROP_LENGTH.equals(name);
  }

  @Override
  public boolean has(final int index, final Scriptable start) {
    return index >= 0 && index < _seq.size() && _seq.get(index) != null;
  }

  @Override
  public void put(final String name, final Scriptable start, final Object value) {
    throw Context.reportRuntimeError("Putting string keys it not allowed for Sequences: " + name);
  }

  @Override
  public void put(final int index, final Scriptable start, final Object value) {
    final Any newElement = AnyWrapper.unwrapAny(value);
    if (index == _seq.size()) {
      _seq.add(newElement);
    } else {
      _seq.set(index, newElement);
    }
  }

  @Override
  public void delete(final String name) {
    throw Context.reportRuntimeError("Deleting string keys is not allowed for Sequences: " + name);
  }

  @Override
  public void delete(final int index) {
    _seq.remove(index);
  }

  @Override
  public Object[] getIds() {
    final Integer[] ids = new Integer[_seq.size()];
    for (int i = 0; i < _seq.size(); i++) {
      ids[i] = i;
    }
    return ids;
  }

}
