/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import java.nio.file.FileVisitResult;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.scripting.ScriptExecutor;
import org.eclipse.smila.scripting.ScriptingEngineException;

/**
 * Evaluate all "*ScriptCatalog.js" files in a directory and merge the results to a single list, sorted by the name of
 * the descriptions. A ScriptCatalog file looks like this:
 *
 * <pre>
 * [
 *   {
 *     "name": "<name of script file and function to expose>",
 *     "description": "<description of function>",
 *     "<property>": "<value>",
 *     ...
 *   },
 *   ...
 * ]
 * </pre>
 *
 * The Visitor adds only those entries to the merged list that have at least a "name" property set. Errors or invalid
 * entries during traversal are ignored.
 */
public class ScriptCatalogVisitor extends SimpleFileVisitor<Path> {
  private static final String JS_SUFFIX = ".js";

  private static final String SCRIPTCATALOG_SUFFIX = "ScriptCatalog" + JS_SUFFIX;

  private final ScriptExecutor _executor;

  private final AnySeq _scripts;

  private final Log _log = LogFactory.getLog(getClass());

  private final Path _scriptDirectory;

  public ScriptCatalogVisitor(final ScriptExecutor executor, final AnySeq scripts, final Path scriptDirectory) {
    _executor = executor;
    _scripts = scripts;
    _scriptDirectory = scriptDirectory;
  }

  @Override
  public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) {
    _log.debug("Visit " + file);
    if (file.getFileName().toString().endsWith(SCRIPTCATALOG_SUFFIX)) {
      try {
        String scriptFile = _scriptDirectory.relativize(file).toString();
        scriptFile = scriptFile.substring(0, scriptFile.length() - JS_SUFFIX.length());
        scriptFile = scriptFile.replace("\\", "/"); // handle windows paths
        _log.info("Reading catalog file " + scriptFile);
        final Object result = _executor.loadScript(scriptFile);
        if (result != null) {
          final Any resultAny = AnyUtil.objectToAny(result);
          for (final Any entry : resultAny) {
            addValidDescription(entry, file);
          }
        }
      } catch (final ScriptingEngineException ex) {
        _log.warn("Cannot read " + file + ", ignoring it. Details: " + ex);
      } catch (final InvalidValueTypeException ex) {
        _log.warn("Cannot convert result from " + file + " to Any object, ignoring it. Details: " + ex);
      }
    }
    return FileVisitResult.CONTINUE;
  }

  private void addValidDescription(final Any entry, final Path file) {
    if (entry.isMap()) {
      final AnyMap entryMap = entry.asMap();
      if (entryMap.containsKey("name")) {
        _scripts.add(entryMap);
      } else {
        _log.warn("Description " + entryMap + " from " + file + " does not contain a name, skipping it.");
      }
    } else {
      _log.warn("Description " + entry + " from " + file + " is not a map, skipping it.");
    }

  }
}
