/**
 *
 */
package org.eclipse.smila.solr;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.lang.NullArgumentException;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.solr.SolrConstants.Errors;
import org.eclipse.smila.utils.config.ConfigUtils;

/**
 * @author pwissel
 *
 */
public class SolrConfig {

  private static final String BUNDLE = "org.eclipse.smila.solr";

  private static final String DEFAULT_CONFIG_FILE = "solr-config.json";

  // General

  private static final String MODE = "mode";

  public static final String ID_FIELDS = "idFields";

  public static final String ID_FIELD_DEFAULT = Record.RECORD_ID;

  public static final String REST_URI = "restUri";

  // ResponseParse

  public static final String RESPONSE_PARSER_ERRORS = "ResponseParser.errors";

  public static final String FETCH_FACET_FIELD_TYPE = "ResponseParser.fetchFacetFieldType";

  // CloudSolrServer

  public static final String ZK_HOST = "CloudSolrServer.zkHost";

  public static final String UPDATES_TO_LEADERS = "CloudSolrServer.updatesToLeaders";

  public static final boolean UPDATES_TO_LEADERS_DEFAULT = true;

  // HttpSolrServer

  public static final String BASE_URL = "HttpSolrServer.baseUrl";

  // EmbeddedSolrServer

  public static final String SOLR_HOME = "EmbeddedSolrServer.solrHome";

  // LoadBalancedSolrServer

  public static final String SERVER_URLS = "LoadBalancedSolrServer.serverUrls";

  private final Log _log = LogFactory.getLog(getClass());

  protected final AnyMap _config;

  public enum Mode {
    CLOUD, EMBEDDED, HTTP;
    public static Mode get(final String label) {
      return valueOf(label.toUpperCase());
    }
  }

  public SolrConfig() {
    this(DEFAULT_CONFIG_FILE);
  }

  public SolrConfig(final String configPath) {
    _config = loadConfiguration(configPath);
  }

  /**
   *
   */
  public SolrConfig(final AnyMap config) {
    _config = config;
  }

  public AnyMap getConfigMap() {
    return _config;
  }

  // General

  public Mode getMode() {
    final String mode = _config.getStringValue(MODE);
    if (mode == null) {
      throw new NullArgumentException(MODE);
    }
    try {
      return Mode.get(mode);
    } catch (final Exception exception) {
      final String message = String.format("Mode must be one of [%s].", StringUtils.join(Mode.values(), ", "));
      throw new IllegalArgumentException(message, exception);
    }
  }

  public String getIdField(final String index) {
    final AnyMap idFields = _config.getMap(ID_FIELDS);
    if (idFields != null) {
      final String id = idFields.getStringValue(index);
      return id != null ? id : ID_FIELD_DEFAULT;
    }
    return ID_FIELD_DEFAULT;
  }

  public String getRestUri() {
    final String restUri = _config.getStringValue(REST_URI);
    if (StringUtils.isBlank(restUri)) {
      throw new NullArgumentException(REST_URI);
    }
    return restUri;
  }

  // Response Parser

  public Errors getResponseParserErrors() {
    return getErrors(RESPONSE_PARSER_ERRORS);
  }

  public boolean isFetchFacetFieldType() {
    final Boolean fetchFacetFieldType = _config.getBooleanValue(FETCH_FACET_FIELD_TYPE);
    return fetchFacetFieldType != null ? fetchFacetFieldType.booleanValue() : false;
  }

  // CloudSolrServer

  public String getCloudSolrServerZkHost() {
    final String zkHost = _config.getStringValue(ZK_HOST);
    if (StringUtils.isBlank(zkHost)) {
      throw new NullArgumentException(ZK_HOST);
    }
    return zkHost;
  }

  public boolean getCloudSolrServerUpdatesToLeaders() {
    final Boolean updatesToLeaders = _config.getBooleanValue(UPDATES_TO_LEADERS);
    return updatesToLeaders != null ? updatesToLeaders.booleanValue() : true;
  }

  // HttpSolrServer

  public String getHttpSolrServerBaseUrl() {
    final String baseUrl = _config.getStringValue(BASE_URL);
    if (StringUtils.isBlank(baseUrl)) {
      throw new NullArgumentException(BASE_URL);
    }
    return baseUrl;
  }

  // EmbeddedSolrServer

  public String getEmbeddedSolrServerSolrHome() {
    final String solrHome = _config.getStringValue(SOLR_HOME);
    if (StringUtils.isBlank(solrHome)) {
      throw new NullArgumentException(SOLR_HOME);
    }
    return solrHome;
  }

  // LoadBalancedSolrServer

  public String[] getLoadBalancedSolrServerServerUrls() {
    final AnySeq serverUrls = _config.getSeq(SERVER_URLS);
    if (serverUrls == null) {
      throw new NullArgumentException(SERVER_URLS);
    }
    return serverUrls.asStrings().toArray(new String[serverUrls.size()]);
  }

  // Internal

  private AnyMap loadConfiguration(final String configPath) {
    final InputStream stream = ConfigUtils.getConfigStream(BUNDLE, configPath);
    final IpcAnyReader reader = new IpcAnyReader();
    Any any = null;
    try {
      any = reader.readJsonStream(stream);
    } catch (IOException exception) {
      if (_log.isFatalEnabled()) {
        _log.fatal("Unable to load Solr configuration.", exception);
      }
    }
    return (AnyMap) any;
  }

  private Errors getErrors(final String clazz) {
    final String errors = _config.getStringValue(clazz);
    try {
      return Errors.get(errors);
    } catch (final Exception exception) {
      return Errors.THROW;
    }
  }

}
