/**
 *
 */
package org.eclipse.smila.solr.query;

import java.util.Date;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.NullArgumentException;
import org.apache.commons.lang.StringUtils;
import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.CommonParams.EchoParamStyle;
import org.apache.solr.common.params.CursorMarkParams;
import org.apache.solr.common.params.FacetParams;
import org.apache.solr.common.params.FacetParams.FacetRangeInclude;
import org.apache.solr.common.params.FacetParams.FacetRangeOther;
import org.apache.solr.common.params.GroupParams;
import org.apache.solr.common.params.HighlightParams;
import org.apache.solr.common.params.MoreLikeThisParams;
import org.apache.solr.common.params.ShardParams;
import org.apache.solr.common.params.SpellingParams;
import org.apache.solr.common.params.TermsParams;
import org.apache.solr.search.QueryParsing;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.util.AnyUtil;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.solr.SolrConstants;
import org.eclipse.smila.solr.SolrConstants.Debug;
import org.eclipse.smila.solr.SolrConstants.FacetMethod;
import org.eclipse.smila.solr.SolrConstants.FacetSort;
import org.eclipse.smila.solr.SolrConstants.GroupFormat;
import org.eclipse.smila.solr.SolrConstants.HighlightBsType;
import org.eclipse.smila.solr.SolrConstants.MoreLikeThisInterestingTerms;
import org.eclipse.smila.solr.SolrConstants.TermsRegexFlag;
import org.eclipse.smila.solr.SolrConstants.TermsSort;
import org.eclipse.smila.solr.SolrUtils;
import org.eclipse.smila.solr.params.QueryParams;
import org.eclipse.smila.solr.params.SolrParams;

/**
 * @author pwissel
 *
 */
public class SolrQueryBuilder extends QueryBuilder {

  /**
   * @param workflowName
   */
  public SolrQueryBuilder() {
    super();
  }

  /**
   * @param workflowName
   * @param request
   */
  public SolrQueryBuilder(Record request) {
    super(request);
  }

  public SolrQueryBuilder setStart(final int start) {
    return (SolrQueryBuilder) setOffset(start);
  }

  public SolrQueryBuilder setRows(final int rows) {
    return (SolrQueryBuilder) setMaxCount(rows);
  }

  public SolrQueryBuilder addFilterQuery(final String fq) {
    return addFilterQuery(fq, null);
  }

  public SolrQueryBuilder addFilterQuery(String fq, final AnyMap localParams) {
    fq = SolrUtils.addLocalParams(fq, localParams).toString();
    return addNative(CommonParams.FQ, fq);
  }

  public SolrQueryBuilder addFilter(final String attribute, final AnyMap filter, final AnyMap localParams) {
    if (localParams != null) {
      filter.put(SolrConstants.LOCAL_PARAMS, localParams);
    }
    return (SolrQueryBuilder) addFilter(attribute, filter);
  }

  public SolrQueryBuilder addReturnField(final String fl) {
    return addReturnField(fl, null);
  }

  public SolrQueryBuilder addReturnField(String fl, final String alias) {
    if (StringUtils.isNotBlank(alias)) {
      fl = alias + QueryStringConstants.COLON + fl;
    }
    return addNative(CommonParams.FL, fl);
  }

  public SolrQueryBuilder setDebug(final Debug debug) {
    if (debug == null) {
      throw new IllegalArgumentException(CommonParams.DEBUG);
    }
    return setNative(CommonParams.DEBUG, debug.toString());
  }

  public SolrQueryBuilder setExplainOther(final String explainOther) {
    return setNative(CommonParams.EXPLAIN_OTHER, explainOther);
  }

  public SolrQueryBuilder setDebugExplainStructured(final boolean debugExplainStructured) {
    return setNative(CommonParams.EXPLAIN_STRUCT, Boolean.toString(debugExplainStructured));
  }

  public SolrQueryBuilder setDefType(final String defType) {
    return setNative(QueryParsing.DEFTYPE, defType);
  }

  public SolrQueryBuilder setTimeAllowed(final long timeAllowed) {
    return setNative(CommonParams.TIME_ALLOWED, String.valueOf(timeAllowed));
  }

  public SolrQueryBuilder setOmitHeader(final boolean omitHeader) {
    return setNative(CommonParams.OMIT_HEADER, Boolean.toString(omitHeader));
  }

  public SolrQueryBuilder setRequestHandler(final String qt) {
    return setNative(CommonParams.QT, qt);
  }

  public SolrQueryBuilder setWriterType(final String wt) {
    return setNative(CommonParams.WT, wt);
  }

  public SolrQueryBuilder setNow(final long ms) {
    return setNative(CommonParams.NOW, String.valueOf(ms));
  }

  public SolrQueryBuilder setTimezone(final String tz) {
    return setNative(CommonParams.TZ, tz);
  }

  public SolrQueryBuilder setEchoHandler(final boolean echoHandler) {
    return setNative(CommonParams.HEADER_ECHO_HANDLER, Boolean.toString(echoHandler));
  }

  public SolrQueryBuilder setEchoParams(final EchoParamStyle echoParams) {
    if (echoParams == null) {
      throw new IllegalArgumentException(CommonParams.HEADER_ECHO_PARAMS);
    }
    return setNative(CommonParams.HEADER_ECHO_PARAMS, echoParams.toString());
  }

  public SolrQueryBuilder setFacet(final boolean facet) {
    return setNative(FacetParams.FACET, Boolean.toString(facet));
  }

  public SolrQueryBuilder addFacetQuery(final String query) {
    return addNative(FacetParams.FACET_QUERY, query);
  }

  public SolrQueryBuilder addFacetByQuery(final String attribute, final String[] expression) {
    return addFacetByQuery(attribute, expression, null, false);
  }

  public SolrQueryBuilder addFacetByQuery(final String attribute, final String[] expression, final AnySeq filter,
    final boolean multiselect) {
    return addFacetByQuery(attribute, expression, filter, multiselect, null);
  }

  public SolrQueryBuilder addFacetByQuery(final String attribute, final String[] expression, final AnySeq filter,
    final boolean multiselect, final AnyMap localParams) {
    if (StringUtils.isBlank(attribute)) {
      throw new NullArgumentException(QueryConstants.ATTRIBUTE);
    }
    if (ArrayUtils.isEmpty(expression)) {
      throw new NullArgumentException(SolrConstants.QUERIES);
    }
    final AnyMap config = getFactory().createAnyMap();
    config.put(SolrConstants.QUERY, attribute);
    config.put(SolrConstants.QUERIES, AnyUtil.objectToAny(expression));
    return addFacetConfig(config, filter, multiselect, localParams);
  }

  public SolrQueryBuilder addFacetByAttribute(final String attribute) {
    return addFacetByAttribute(attribute, null, false);
  }

  public SolrQueryBuilder addFacetByAttribute(final String attribute, final AnySeq filter, final boolean multiselect) {
    return addFacetByAttribute(attribute, filter, multiselect, null);
  }

  public SolrQueryBuilder addFacetByAttribute(final String attribute, final AnySeq filter,
    final boolean multiselect, final AnyMap localParams) {
    // create facet and add attribute
    if (StringUtils.isBlank(attribute)) {
      throw new NullArgumentException(QueryConstants.ATTRIBUTE);
    }
    final AnyMap config = getFactory().createAnyMap();
    config.put(QueryConstants.ATTRIBUTE, attribute);
    return addFacetConfig(config, filter, multiselect, localParams);
  }

  public SolrQueryBuilder setFacetPrefix(final String prefix) {
    return setFacetPrefix(prefix, null);
  }

  public SolrQueryBuilder setFacetPrefix(final String prefix, final String fieldName) {
    return setNative(FacetParams.FACET_PREFIX, prefix, fieldName);
  }

  public SolrQueryBuilder setFacetSort(final FacetSort facetSort) {
    return setFacetSort(facetSort, null);
  }

  public SolrQueryBuilder setFacetSort(final FacetSort sort, final String fieldName) {
    if (sort == null) {
      throw new IllegalArgumentException(FacetParams.FACET_SORT);
    }
    return setNative(FacetParams.FACET_SORT, sort.toString(), fieldName);
  }

  public SolrQueryBuilder setFacetLimit(final int limit) {
    return setFacetLimit(limit, null);
  }

  public SolrQueryBuilder setFacetLimit(final int limit, final String fieldName) {
    return setNative(FacetParams.FACET_LIMIT, String.valueOf(limit), fieldName);
  }

  public SolrQueryBuilder setFacetOffset(final int offset) {
    return setFacetOffset(offset, null);
  }

  public SolrQueryBuilder setFacetOffset(final int offset, final String fieldName) {
    return setNative(FacetParams.FACET_OFFSET, String.valueOf(offset), fieldName);
  }

  public SolrQueryBuilder setFacetMincount(final int mincount) {
    return setFacetMincount(mincount, null);
  }

  public SolrQueryBuilder setFacetMincount(final int mincount, final String fieldName) {
    return setNative(FacetParams.FACET_MINCOUNT, String.valueOf(mincount), fieldName);
  }

  public SolrQueryBuilder setFacetMissing(final boolean missing) {
    return setFacetMissing(missing, null);
  }

  public SolrQueryBuilder setFacetMissing(final boolean missing, final String fieldName) {
    return setNative(FacetParams.FACET_MISSING, Boolean.toString(missing), fieldName);
  }

  public SolrQueryBuilder setFacetMethod(final FacetMethod method) {
    return setFacetMethod(method, null);
  }

  public SolrQueryBuilder setFacetMethod(final FacetMethod facetMethod, final String fieldName) {
    if (facetMethod == null) {
      throw new IllegalArgumentException(FacetParams.FACET_METHOD);
    }
    return setNative(FacetParams.FACET_METHOD, facetMethod.toString(), fieldName);
  }

  public SolrQueryBuilder setFacetEnumCacheMinDf(final int enumCacheMinDf) {
    return setFacetEnumCacheMinDf(enumCacheMinDf, null);
  }

  public SolrQueryBuilder setFacetEnumCacheMinDf(final int enumCacheMinDf, final String fieldName) {
    return setNative(FacetParams.FACET_ENUM_CACHE_MINDF, String.valueOf(enumCacheMinDf), fieldName);
  }

  public SolrQueryBuilder setFacetThreads(final int threads) {
    return setNative(FacetParams.FACET_THREADS, String.valueOf(threads), null);
  }

  public SolrQueryBuilder addFacetByRange(final String range, final Number start, final Number end, final Number gap) {
    return addFacetByRange(range, start, end, gap, null);
  }

  public SolrQueryBuilder addFacetByRange(final String range, final Number start, final Number end,
    final Number gap, final String outputKey) {
    return addFacetByRange(range, start, end, gap, outputKey, false);
  }

  public SolrQueryBuilder addFacetByRange(final String range, final Number start, final Number end,
    final Number gap, final String outputKey, final boolean multiselect) {
    if (StringUtils.isBlank(range)) {
      throw new IllegalArgumentException(SolrConstants.RANGE);
    }
    final AnyMap rangeFacet = getFactory().createAnyMap();
    rangeFacet.put(SolrConstants.RANGE, range);
    rangeFacet.put(SolrConstants.START, start);
    rangeFacet.put(SolrConstants.END, end);
    rangeFacet.put(SolrConstants.GAP, gap);
    if (StringUtils.isNotBlank(outputKey)) {
      rangeFacet.put(QueryStringConstants.KEY, outputKey);
    }
    if (multiselect) {
      rangeFacet.put(SolrConstants.MULTISELECT, multiselect);
    }
    return addConfig(QueryConstants.FACETBY, rangeFacet);
  }

  public SolrQueryBuilder addFacetByRange(final String range, final Date start, final Date end, final String gap) {
    return addFacetByRange(range, start, end, gap, null);
  }

  public SolrQueryBuilder addFacetByRange(final String range, final Date start, final Date end, final String gap,
    final String outputKey) {
    return addFacetByRange(range, start, end, gap, outputKey, false);
  }

  public SolrQueryBuilder addFacetByRange(final String range, final Date start, final Date end, final String gap,
    final String outputKey, final boolean multiselect) {
    if (StringUtils.isBlank(range)) {
      throw new IllegalArgumentException(SolrConstants.RANGE);
    }
    final AnyMap rangeFacet = getFactory().createAnyMap();
    rangeFacet.put(SolrConstants.RANGE, range);
    rangeFacet.put(SolrConstants.START, AnyUtil.objectToAny(start));
    rangeFacet.put(SolrConstants.END, AnyUtil.objectToAny(end));
    // TODO: can gap be null?
    rangeFacet.put(SolrConstants.GAP, AnyUtil.objectToAny(gap));
    if (StringUtils.isNotBlank(outputKey)) {
      rangeFacet.put(QueryStringConstants.KEY, outputKey);
    }
    if (multiselect) {
      rangeFacet.put(SolrConstants.MULTISELECT, multiselect);
    }
    return addConfig(QueryConstants.FACETBY, rangeFacet);
  }

  public SolrQueryBuilder addFacetPivot(final String... pivot) {
    if (pivot == null) {
      throw new IllegalArgumentException(SolrConstants.PIVOT);
    }
    final AnyMap pivotFacet = getFactory().createAnyMap();
    final Any values = AnyUtil.objectToAny(pivot);
    if (values.isSeq()) {
      pivotFacet.getSeq(SolrConstants.PIVOT, true).addAll(values.asSeq());
    }
    return addConfig(QueryConstants.FACETBY, pivotFacet);
  }

  public SolrQueryBuilder addFacetInterval(final String interval, final String... set) {
    if (StringUtils.isBlank(interval)) {
      throw new IllegalArgumentException(SolrConstants.INTERVAL);
    }
    final AnyMap intervalFacet = getFactory().createAnyMap();
    intervalFacet.put(SolrConstants.INTERVAL, interval);
    final Any values = AnyUtil.objectToAny(set);
    if (values.isSeq()) {
      intervalFacet.getSeq(SolrConstants.SET, true).addAll(values.asSeq());
    }
    return addConfig(QueryConstants.FACETBY, intervalFacet);
  }

  public SolrQueryBuilder setFacetRangeHardend(final boolean hardend) {
    return setFacetRangeHardend(hardend, null);
  }

  public SolrQueryBuilder setFacetRangeHardend(final boolean hardend, final String fieldName) {
    return setNative(FacetParams.FACET_RANGE_HARD_END, Boolean.toString(hardend), fieldName);
  }

  public SolrQueryBuilder setFacetRangeOther(final FacetRangeOther other) {
    return setFacetRangeOther(other, null);
  }

  public SolrQueryBuilder setFacetRangeOther(final FacetRangeOther other, final String fieldName) {
    if (other == null) {
      throw new IllegalArgumentException(FacetParams.FACET_RANGE_OTHER);
    }
    return setNative(FacetParams.FACET_RANGE_OTHER, other.toString(), fieldName);
  }

  public SolrQueryBuilder setFacetRangeInclude(final FacetRangeInclude include) {
    return setFacetRangeInclude(include, null);
  }

  public SolrQueryBuilder setFacetRangeInclude(final FacetRangeInclude include, final String fieldName) {
    if (include == null) {
      throw new IllegalArgumentException(FacetParams.FACET_RANGE_INCLUDE);
    }
    return setNative(FacetParams.FACET_RANGE_INCLUDE, include.toString(), fieldName);
  }

  public SolrQueryBuilder setFacetPivotMincount(final int mincount) {
    return setNative(FacetParams.FACET_PIVOT_MINCOUNT, String.valueOf(mincount));
  }

  public SolrQueryBuilder setGroup(final boolean group) {
    return setNative(GroupParams.GROUP, Boolean.toString(group));
  }

  public SolrQueryBuilder addGroupByAttribute(final String attribute) {
    if (StringUtils.isBlank(attribute)) {
      throw new IllegalArgumentException(GroupParams.GROUP_FIELD);
    }
    final AnyMap group = getFactory().createAnyMap();
    group.put(QueryConstants.ATTRIBUTE, attribute);
    return addConfig(QueryConstants.GROUPBY, group);
  }

  public SolrQueryBuilder addGroupByFunctionQuery(final String func) {
    if (StringUtils.isBlank(func)) {
      throw new IllegalArgumentException(GroupParams.GROUP_FUNC);
    }
    final AnyMap group = getFactory().createAnyMap();
    group.put(SolrConstants.FUNC, func);
    return addConfig(QueryConstants.GROUPBY, group);
  }

  public SolrQueryBuilder addGroupByQuery(final String query) {
    if (StringUtils.isBlank(query)) {
      throw new IllegalArgumentException(GroupParams.GROUP_QUERY);
    }
    final AnyMap group = getFactory().createAnyMap();
    group.put(SolrConstants.QUERY, query);
    return addConfig(QueryConstants.GROUPBY, group);
  }

  public SolrQueryBuilder setGroupLimit(final int limit) {
    return setNative(GroupParams.GROUP_LIMIT, String.valueOf(limit));
  }

  public SolrQueryBuilder setGroupOffset(final int offset) {
    return setNative(GroupParams.GROUP_OFFSET, String.valueOf(offset));
  }

  public SolrQueryBuilder setGroupFormat(final GroupFormat format) {
    if (format == null) {
      throw new IllegalArgumentException(GroupParams.GROUP_FORMAT);
    }
    return setNative(GroupParams.GROUP_FORMAT, format.toString());
  }

  public SolrQueryBuilder setGroupMain(final boolean main) {
    return setNative(GroupParams.GROUP_MAIN, Boolean.toString(main));
  }

  public SolrQueryBuilder setGroupNGroups(final boolean ngroups) {
    return setNative(GroupParams.GROUP_TOTAL_COUNT, Boolean.toString(ngroups));
  }

  public SolrQueryBuilder setGroupTruncate(final boolean truncate) {
    return setNative(GroupParams.GROUP_TRUNCATE, Boolean.toString(truncate));
  }

  public SolrQueryBuilder setGroupFacet(final boolean facet) {
    return setNative(GroupParams.GROUP_FACET, Boolean.toString(facet));
  }

  public SolrQueryBuilder setGroupCachePercent(final int cachePercent) {
    if (cachePercent < 0 || cachePercent > 100) {
      throw new IllegalArgumentException(GroupParams.GROUP_CACHE_PERCENTAGE);
    }
    return setNative(GroupParams.GROUP_CACHE_PERCENTAGE, String.valueOf(cachePercent));
  }

  public SolrQueryBuilder setHighlight(final boolean highlight) {
    return setNative(HighlightParams.HIGHLIGHT, Boolean.toString(highlight));
  }

  public SolrQueryBuilder setHighlightQuery(final String query) {
    return setNative(HighlightParams.Q, query);
  }

  public SolrQueryBuilder addHighlightField(final String field) {
    if (StringUtils.isBlank(field)) {
      throw new IllegalArgumentException(HighlightParams.FIELDS);
    }
    final AnyMap config = getFactory().createAnyMap();
    config.put(QueryConstants.ATTRIBUTE, field);
    return addConfig(QueryConstants.HIGHLIGHT, config);
  }

  public SolrQueryBuilder setHighlightSnippets(final int snippets) {
    return setHighlightSnippets(snippets, null);
  }

  public SolrQueryBuilder setHighlightSnippets(final int snippets, final String fieldName) {
    return setNative(HighlightParams.SNIPPETS, String.valueOf(snippets), fieldName);
  }

  public SolrQueryBuilder setHighlightFragsize(final int fragsize) {
    return setHighlightFragsize(fragsize, null);
  }

  public SolrQueryBuilder setHighlightFragsize(final int fragsize, final String fieldName) {
    return setNative(HighlightParams.FRAGSIZE, String.valueOf(fragsize), fieldName);
  }

  public SolrQueryBuilder setHighlightMergeContiguous(final boolean mergeContiguous) {
    return setHighlightMergeContiguous(mergeContiguous, null);
  }

  public SolrQueryBuilder setHighlightMergeContiguous(final boolean mergeContiguous, final String fieldName) {
    return setNative(HighlightParams.MERGE_CONTIGUOUS_FRAGMENTS, Boolean.toString(mergeContiguous), fieldName);
  }

  public SolrQueryBuilder setHighlightRequireFieldMatch(final boolean requireFieldMatch) {
    return setNative(HighlightParams.FIELD_MATCH, Boolean.toString(requireFieldMatch));
  }

  public SolrQueryBuilder setHighlightMaxAnalysedChars(final int maxAnalysedChars) {
    return setNative(HighlightParams.MAX_CHARS, String.valueOf(maxAnalysedChars));
  }

  public SolrQueryBuilder setHighlightAlternateField(final String alternateField) {
    return setHighlightAlternateField(alternateField, null);
  }

  public SolrQueryBuilder setHighlightAlternateField(final String alternateField, final String fieldName) {
    return setNative(HighlightParams.ALTERNATE_FIELD, alternateField, fieldName);
  }

  public SolrQueryBuilder setHighlightAlternateFieldLength(final int alternateFieldLength) {
    return setNative(HighlightParams.ALTERNATE_FIELD_LENGTH, String.valueOf(alternateFieldLength));
  }

  public SolrQueryBuilder setHighlightPreserveMulti(final boolean preserveMulti) {
    return setNative(HighlightParams.PRESERVE_MULTI, Boolean.toString(preserveMulti));
  }

  public SolrQueryBuilder setHighlightMaxMultiValuedToExamine(final int maxMultiValuedToExamine) {
    return setNative(HighlightParams.MAX_MULTIVALUED_TO_EXAMINE, String.valueOf(maxMultiValuedToExamine), null);
  }

  public SolrQueryBuilder setHighlightMaxMultiValuedToMatch(final int maxMultiValuedToMatch) {
    return setNative(HighlightParams.MAX_MULTIVALUED_TO_MATCH, String.valueOf(maxMultiValuedToMatch), null);
  }

  public SolrQueryBuilder setHighlightFormatter(final String formatter) {
    return setHighlightFormatter(formatter, null);
  }

  public SolrQueryBuilder setHighlightFormatter(final String formatter, final String fieldName) {
    return setNative(HighlightParams.FORMATTER, formatter, fieldName);
  }

  public SolrQueryBuilder setHighlightSimplePre(final String simplePre) {
    return setHighlightSimplePre(simplePre, null);
  }

  public SolrQueryBuilder setHighlightSimplePre(final String simplePre, final String fieldName) {
    return setNative(HighlightParams.SIMPLE_PRE, simplePre, fieldName);
  }

  public SolrQueryBuilder setHighlightSimplePost(final String simplePost) {
    return setHighlightSimplePost(simplePost, null);
  }

  public SolrQueryBuilder setHighlightSimplePost(final String simplePost, final String fieldName) {
    return setNative(HighlightParams.SIMPLE_POST, simplePost, fieldName);
  }

  public SolrQueryBuilder setHighlightFragmeter(final String fragmenter) {
    return setHighlightFragmenter(fragmenter, null);
  }

  public SolrQueryBuilder setHighlightFragmenter(final String fragmenter, final String fieldName) {
    return setNative(HighlightParams.FRAGMENTER, fragmenter, fieldName);
  }

  public SolrQueryBuilder setHighlightFragListBuilder(final String fragListBuilder) {
    return setHighlightFragListBuilder(fragListBuilder, null);
  }

  public SolrQueryBuilder setHighlightFragListBuilder(final String fragListBuilder, final String fieldName) {
    return setNative(HighlightParams.FRAG_LIST_BUILDER, fragListBuilder, fieldName);
  }

  public SolrQueryBuilder setHighlightFragmentsBuilder(final String fragmentsBuilder) {
    return setNative(HighlightParams.FRAGMENTS_BUILDER, fragmentsBuilder);
  }

  public SolrQueryBuilder setHighlightBoundaryScanner(final String boundaryScanner) {
    return setNative(HighlightParams.BOUNDARY_SCANNER, boundaryScanner);
  }

  public SolrQueryBuilder setHighlightBsMaxScan(final int bsMaxScan) {
    return setNative(HighlightParams.BS_MAX_SCAN, String.valueOf(bsMaxScan));
  }

  public SolrQueryBuilder setHighlightBsChars(final String bsChars) {
    return setNative(HighlightParams.BS_CHARS, bsChars);
  }

  public SolrQueryBuilder setHighlightBsType(final HighlightBsType bsType) {
    return setNative(HighlightParams.BS_TYPE, bsType.toString());
  }

  public SolrQueryBuilder setHighlightBsLanguage(final String bsLanguage) {
    return setNative(HighlightParams.BS_LANGUAGE, bsLanguage);
  }

  public SolrQueryBuilder setHighlightBsCountry(final String bsCountry) {
    return setNative(HighlightParams.BS_COUNTRY, bsCountry);
  }

  public SolrQueryBuilder setHighlightUseFastVectorHighlighter(final boolean useFastVectorHighlighter) {
    return setHighlightUseFastVectorHighlighter(useFastVectorHighlighter, null);
  }

  public SolrQueryBuilder setHighlightUseFastVectorHighlighter(final boolean useFastVectorHighlighter,
    final String fieldName) {
    return setNative(HighlightParams.USE_FVH, Boolean.toString(useFastVectorHighlighter), fieldName);
  }

  public SolrQueryBuilder setHighlightUsePhraseHighlighter(final boolean usePhaseHighlighter) {
    return setNative(HighlightParams.USE_PHRASE_HIGHLIGHTER, Boolean.toString(usePhaseHighlighter));
  }

  public SolrQueryBuilder setHighlightMultiTerm(final boolean highlightMultiTerm) {
    return setNative(HighlightParams.HIGHLIGHT_MULTI_TERM, Boolean.toString(highlightMultiTerm));
  }

  public SolrQueryBuilder setHighlightRegexSlop(final float regexSlop) {
    return setNative(HighlightParams.SLOP, String.valueOf(regexSlop));
  }

  public SolrQueryBuilder setHighlightRegexPattern(final String regexPattern) {
    return setNative(HighlightParams.PATTERN, regexPattern);
  }

  public SolrQueryBuilder setHighlightRegexMaxAnalyzedChars(final int regexMaxAnalyzedChars) {
    return setNative(HighlightParams.MAX_RE_CHARS, String.valueOf(regexMaxAnalyzedChars));
  }

  public SolrQueryBuilder setMoreLikeThis(final boolean mlt) {
    return setNative(MoreLikeThisParams.MLT, Boolean.toString(mlt));
  }

  public SolrQueryBuilder setMoreLikeThisSimilarityFields(final String[] fl) {
    return setNative(MoreLikeThisParams.SIMILARITY_FIELDS, StringUtils.join(fl, QueryStringConstants.COMMA));
  }

  public SolrQueryBuilder setMoreLikeThisCount(final int count) {
    return setNative(MoreLikeThisParams.DOC_COUNT, String.valueOf(count));
  }

  public SolrQueryBuilder setMoreLikeThisMinTermFrequency(final int minTermFrequency) {
    return setNative(MoreLikeThisParams.MIN_TERM_FREQ, String.valueOf(minTermFrequency));
  }

  public SolrQueryBuilder setMoreLikeThisMinDocumentFrequency(final int minDocumentFrequenxy) {
    return setNative(MoreLikeThisParams.MIN_DOC_FREQ, String.valueOf(minDocumentFrequenxy));
  }

  public SolrQueryBuilder setMoreLikeThisMinWordLength(final int minWordLength) {
    return setNative(MoreLikeThisParams.MIN_WORD_LEN, String.valueOf(minWordLength));
  }

  public SolrQueryBuilder setMoreLikeThisMaxWordLength(final int maxWordLength) {
    return setNative(MoreLikeThisParams.MAX_WORD_LEN, String.valueOf(maxWordLength));
  }

  public SolrQueryBuilder setMoreLikeThisMaxQueryTerms(final int maxQueryTerms) {
    return setNative(MoreLikeThisParams.MAX_QUERY_TERMS, String.valueOf(maxQueryTerms));
  }

  public SolrQueryBuilder setMoreLikeThisMaxTokens(final int maxTokens) {
    return setNative(MoreLikeThisParams.MAX_NUM_TOKENS_PARSED, String.valueOf(maxTokens));
  }

  public SolrQueryBuilder setMoreLikeThisBoost(final boolean boost) {
    return setNative(MoreLikeThisParams.BOOST, Boolean.toString(boost));
  }

  public SolrQueryBuilder setMoreLikeThisQueryFields(final String[] qf) {
    return setNative(MoreLikeThisParams.QF, StringUtils.join(qf, QueryStringConstants.WHITESPACE));
  }

  public SolrQueryBuilder setMoreLikeThisMatchInclude(final boolean matchInclude) {
    return setNative(MoreLikeThisParams.MATCH_INCLUDE, Boolean.toString(matchInclude));
  }

  public SolrQueryBuilder setMoreLikeThisMatchOffset(final int matchOffset) {
    return setNative(MoreLikeThisParams.MATCH_OFFSET, String.valueOf(matchOffset));
  }

  public SolrQueryBuilder setMoreLikeThisInterestingTerms(final MoreLikeThisInterestingTerms interestingTerms) {
    if (interestingTerms == null) {
      throw new IllegalArgumentException(MoreLikeThisParams.INTERESTING_TERMS);
    }
    return setNative(MoreLikeThisParams.INTERESTING_TERMS, interestingTerms.toString());
  }

  public SolrQueryBuilder setSpellcheck(final boolean spellcheck) {
    return setNative(SolrConstants.SPELLCHECK, Boolean.toString(spellcheck));
  }

  public SolrQueryBuilder setSpellcheckQuery(final String query) {
    return setNative(SpellingParams.SPELLCHECK_Q, query);
  }

  public SolrQueryBuilder setSpellcheckBuild(final boolean build) {
    return setNative(SpellingParams.SPELLCHECK_BUILD, Boolean.toString(build));
  }

  public SolrQueryBuilder setSpellcheckReload(final boolean reload) {
    return setNative(SpellingParams.SPELLCHECK_RELOAD, Boolean.toString(reload));
  }

  public SolrQueryBuilder setSpellcheckDictionary(final String dictionary) {
    return setNative(SpellingParams.SPELLCHECK_DICT, dictionary);
  }

  public SolrQueryBuilder setSpellcheckCount(final int count) {
    return setNative(SpellingParams.SPELLCHECK_COUNT, String.valueOf(count));
  }

  public SolrQueryBuilder setSpellcheckAlternativeTermCount(final int alternativeTermCount) {
    return setNative(SpellingParams.SPELLCHECK_ALTERNATIVE_TERM_COUNT, String.valueOf(alternativeTermCount));
  }

  public SolrQueryBuilder setSpellchekOnlyMorePopular(final boolean onlyMorePopular) {
    return setNative(SpellingParams.SPELLCHECK_ONLY_MORE_POPULAR, Boolean.toString(onlyMorePopular));
  }

  public SolrQueryBuilder setSpellcheckMaxResultsForSuggest(final int maxResultsForSuggest) {
    return setNative(SpellingParams.SPELLCHECK_MAX_RESULTS_FOR_SUGGEST, String.valueOf(maxResultsForSuggest));
  }

  public SolrQueryBuilder setSpellcheckExtendedResults(final boolean extendedResults) {
    return setNative(SpellingParams.SPELLCHECK_EXTENDED_RESULTS, Boolean.toString(extendedResults));
  }

  public SolrQueryBuilder setSpellcheckCollate(final boolean coolate) {
    return setNative(SpellingParams.SPELLCHECK_COLLATE, Boolean.toString(coolate));
  }

  public SolrQueryBuilder setSpellcheckMaxCollations(final int maxCollations) {
    return setNative(SpellingParams.SPELLCHECK_MAX_COLLATIONS, String.valueOf(maxCollations));
  }

  public SolrQueryBuilder setSpellcheckMaxCollationTries(final int maxCollationTries) {
    return setNative(SpellingParams.SPELLCHECK_MAX_COLLATION_TRIES, String.valueOf(maxCollationTries));
  }

  public SolrQueryBuilder setSpellcheckMaxCollationEvaluations(final int maxCollationEvaluations) {
    return setNative(SpellingParams.SPELLCHECK_MAX_COLLATION_EVALUATIONS, String.valueOf(maxCollationEvaluations));
  }

  public SolrQueryBuilder setSpellcheckCollateParam(final String collateParam, final String collateParamValue) {
    final String parameterName = SpellingParams.SPELLCHECK_COLLATE_PARAM_OVERRIDE + collateParam;
    return setNative(parameterName, collateParamValue);
  }

  public SolrQueryBuilder setSpellcheckCollateExtendedResults(final boolean collateExtendedResults) {
    return setNative(SpellingParams.SPELLCHECK_COLLATE_EXTENDED_RESULTS, String.valueOf(collateExtendedResults));
  }

  public SolrQueryBuilder setSpellcheckCollateMaxCollectDocs(final int collateMaxcollateDocs) {
    return setNative(SpellingParams.SPELLCHECK_COLLATE_MAX_COLLECT_DOCS, String.valueOf(collateMaxcollateDocs));
  }

  public SolrQueryBuilder setSpellcheckAccuracy(final float accuracy) {
    return setNative(SpellingParams.SPELLCHECK_ACCURACY, String.valueOf(accuracy));
  }

  public SolrQueryBuilder setSpellcheckDictionaryKey(final String dictionary, final String key, final String value) {
    final String parameterName = SpellingParams.SPELLCHECK_PREFIX + dictionary + QueryStringConstants.PERIOT + key;
    return setNative(parameterName, value);
  }

  public SolrQueryBuilder setTerms(final boolean terms) {
    return setNative(TermsParams.TERMS, Boolean.toString(terms));
  }

  public SolrQueryBuilder addTermsField(final String field) {
    return setNative(TermsParams.TERMS_FIELD, field);
  }

  public SolrQueryBuilder setTermsLower(final String lower) {
    return setNative(TermsParams.TERMS_LOWER, lower);
  }

  public SolrQueryBuilder setTermsLowerInclusive(final boolean lowerInclusive) {
    return setNative(TermsParams.TERMS_LOWER_INCLUSIVE, Boolean.toString(lowerInclusive));
  }

  public SolrQueryBuilder setTermsMincount(final int mincount) {
    return setNative(TermsParams.TERMS_MINCOUNT, String.valueOf(mincount));
  }

  public SolrQueryBuilder setTermsMaxcount(final int maxcount) {
    return setNative(TermsParams.TERMS_MAXCOUNT, String.valueOf(maxcount));
  }

  public SolrQueryBuilder setTermsPrefix(final String prefix) {
    return setNative(TermsParams.TERMS_PREFIX_STR, prefix);
  }

  public SolrQueryBuilder setTermsRegex(final String regex) {
    return setNative(TermsParams.TERMS_REGEXP_STR, regex);
  }

  public SolrQueryBuilder addTermsRegexFlag(final TermsRegexFlag regexFlag) {
    if (regexFlag == null) {
      throw new IllegalArgumentException(TermsParams.TERMS_REGEXP_FLAG);
    }
    return setNative(TermsParams.TERMS_REGEXP_FLAG, regexFlag.toString());
  }

  public SolrQueryBuilder setTermsLimit(final int limit) {
    return setNative(TermsParams.TERMS_LIMIT, String.valueOf(limit));
  }

  public SolrQueryBuilder setTermsUpper(final String upper) {
    return setNative(TermsParams.TERMS_UPPER, upper);
  }

  public SolrQueryBuilder setTermsUpperInclusive(final boolean upperInclusive) {
    return setNative(TermsParams.TERMS_UPPER_INCLUSIVE, Boolean.toString(upperInclusive));
  }

  public SolrQueryBuilder setTermsRaw(final boolean raw) {
    return setNative(TermsParams.TERMS_RAW, Boolean.toString(raw));
  }

  public SolrQueryBuilder setTermsSort(final TermsSort sort) {
    return setNative(TermsParams.TERMS_SORT, sort.toString());
  }

  public SolrQueryBuilder setShards(final String... shards) {
    return setNative(ShardParams.SHARDS, StringUtils.join(shards, QueryStringConstants.COMMA));
  }

  public SolrQueryBuilder setShardsRequestHandler(final String qt) {
    return setNative(ShardParams.SHARDS_QT, qt);
  }

  public SolrQueryBuilder setShardsInfo(final boolean info) {
    return setNative(ShardParams.SHARDS_INFO, Boolean.toString(info));
  }

  public SolrQueryBuilder setShardsTolerant(final boolean tolerant) {
    return setNative(ShardParams.SHARDS_TOLERANT, Boolean.toString(tolerant));
  }

  public SolrQueryBuilder setCursorMarkStart() {
    return setCursorMark(CursorMarkParams.CURSOR_MARK_START);
  }

  public SolrQueryBuilder setCursorMark(final String cursorMark) {
    return setNative(CursorMarkParams.CURSOR_MARK_PARAM, cursorMark);
  }

  public SolrQueryBuilder setCollection(final String... collection) {
    return setNative(SolrConstants.COLLECTION, StringUtils.join(collection, QueryStringConstants.COMMA));
  }

  public SolrQueryBuilder setRoute(final String route) {
    return setNative(SolrConstants.ROUTE, route);
  }

  public SolrQueryBuilder setQueryFindAll() {
    return (SolrQueryBuilder) setQuery(QueryStringConstants.WILDCARD + QueryStringConstants.COLON
      + QueryStringConstants.WILDCARD);
  }

  public SolrQueryBuilder setNative(String parameterName, final String value) {
    return setNative(parameterName, value, null);
  }

  public SolrQueryBuilder setNative(String parameterName, final String value, final String fieldName) {
    parameterName = getParameterName(parameterName, fieldName);
    if (value != null) {
      // set mapping in query
      getNativeParams().put(parameterName, value);
    } else {
      // remove mapping from query
      getNativeParams().remove(parameterName);
    }
    return this;
  }

  public SolrQueryBuilder addNative(String parameterName, final String value) {
    return addNative(parameterName, value, null);
  }

  public SolrQueryBuilder addNative(String parameterName, final String value, final String fieldName) {
    parameterName = getParameterName(parameterName, fieldName);
    if (value != null) {
      final Value val = getFactory().createStringValue(value);
      getNativeParams().add(parameterName, val);
    }
    return this;
  }

  SolrQueryBuilder addConfig(final String parameterName, final AnyMap config) {
    getMetadata().getSeq(parameterName, true).add(config);
    return this;
  }

  private String getParameterName(String parameterName, final String fieldName) {
    if (StringUtils.isBlank(parameterName)) {
      throw new IllegalArgumentException("parameter name must not be empty or null.");
    }
    if (fieldName != null) {
      return SolrUtils.getPerFieldParameter(fieldName, parameterName);
    }
    return parameterName;
  }

  private AnyMap getNativeParams() {
    return getMetadata().getMap(SolrParams.SOLR_PARAMETER_ATTRIBUTE, true).getMap(QueryParams.QUERY, true);
  }

  private SolrQueryBuilder addFacetConfig(final AnyMap config, final AnySeq filter, final boolean multiselect,
    final AnyMap localParams) {
    // add filter and multiselect
    if (filter != null) {
      config.put(QueryConstants.FILTER, filter);
    }
    if (multiselect) {
      config.put(SolrConstants.MULTISELECT, multiselect);
    }
    // add local params
    if (localParams != null) {
      config.put(SolrConstants.LOCAL_PARAMS, localParams);
    }
    return addConfig(QueryConstants.FACETBY, config);
  }

}
