/*******************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.server.util;

import java.io.IOException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.HttpStatus;

public final class CorsUtils {
  /**
   * Turn on Cross Origin Resource Sharing (see http://www.w3.org/TR/cors/) to allow web applications from other
   * servers/ports to communicate with us. Handle OPTIONS requests inline.
   *
   * @return true if the request should not be forwarded to the actual handler anymore, but was handled completely here
   *         (for OPTIONS requests, usually).
   * @throws IOException
   */
  public static boolean setCorsResponseHandlers(final HttpExchange he) throws IOException {
    String origin = he.getRequestHeader("Origin");
    if (origin == null) {
      origin = "*";
    }
    he.setResponseHeader("Access-Control-Allow-Origin", origin);
    he.setResponseHeader("Access-Control-Allow-Credentials", "true");
    if ("OPTIONS".equals(he.getRequestMethod())) {
      // Handle CORS OPTIONS transparent for handler
      final String requestedMethod = he.getRequestHeader("Access-Control-Request-Method");
      if (requestedMethod != null) {
        // Allow all requested methods
        he.setResponseHeader("Access-Control-Allow-Methods", requestedMethod);

        // Set cache timeout to 20 days
        he.setResponseHeader("Access-Control-Max-Age", "1728000");

        // Allow all custom headers
        final String customHeaders = he.getRequestHeader("Access-Control-Request-Headers");
        if (customHeaders != null) {
          // JQuery is setting "x-requested-with" for example
          he.setResponseHeader("Access-Control-Allow-Headers", customHeaders);
        }
        he.setResponseStatus(HttpStatus.OK);
        he.getResponseStream().flush();
        return true;
      }
    }
    return false;
  }

  /**
   * Turn on Cross Origin Resource Sharing (see http://www.w3.org/TR/cors/) to allow web applications from other
   * servers/ports to communicate with us.
   *
   * @param request
   * @param response
   * @throws IOException
   */
  public static void setCorsResponseHandlers(final HttpServletRequest request, final HttpServletResponse response)
    throws IOException {
    String origin = request.getHeader("Origin");
    if (origin == null) {
      origin = "*";
    }
    response.setHeader("Access-Control-Allow-Origin", origin);
    response.setHeader("Access-Control-Allow-Credentials", "true");
    response.setHeader("Access-Control-Expose-Headers",
      "WWW-Authenticate,Access-Control-Allow-Origin,Access-Control-Allow-Credentials");
  }
}
