/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Tobias Liefke - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.pipelets;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Map.Entry;

import javax.script.Bindings;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;

/**
 * Executes a JavaScript on the meta data of every record.
 * 
 * @author Tobias Liefke
 */
public class ScriptPipelet implements Pipelet {

  /** Name of the property that contains the script. */
  public static final String PROPERTY_SCRIPT = "script";

  /** Name of the property that contains the script file. */
  public static final String PROPERTY_SCRIPT_FILE = "scriptFile";

  /** Name of the property that contains the script language. */
  public static final String PROPERTY_TYPE = "type";

  /** Name of the property that will receive the result of the script. */
  public static final String PROPERTY_RESULT_ATTRIBUTE = "resultAttribute";

  /** The name of the script variable that contains the blackboard. */
  private static final String BLACKBOARD_VARIABLE = "blackboard";

  /** The name of the script variable that contains the id of the current record. */
  private static final String ID_VARIABLE = "id";

  /** The name of the script variable that contains the metadata of the current record. */
  private static final String METADATA_VARIABLE = "record";

  /** The name of the script variable that contains a wrapper for the result collector. */
  private static final String RESULTS_VARIABLE = "results";

  /** the name of the script variable that contains the pipelet configuration. */
  private static final String PARAMETER_ACCESSOR_VARIABLE = "parameterAccessor";

  /** The local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** The pipelet configuration. */
  private AnyMap _configuration;

  /** The configured script file. */
  private File _scriptFile;

  /** The last known modification time of the script file. */
  private long _fileLastModified;

  /** The configured script. */
  private String _script;

  /** The current engine. */
  private ScriptEngine _engine;

  /** Used to create engines. */
  private ScriptEngineManager _engines;

  /**
   * {@inheritDoc}
   */
  @Override
  public void configure(final AnyMap configuration) throws ProcessingException {
    _configuration = configuration;
    final ParameterAccessor paramAccessor = new ParameterAccessor(null, configuration);

    // Read script file
    final String scriptFile = paramAccessor.getParameter(PROPERTY_SCRIPT_FILE, null);
    if (scriptFile != null) {
      _scriptFile = new File(scriptFile);
      if (!_scriptFile.isFile()) {
        throw new ProcessingException("Could not find script file: " + _scriptFile.getAbsolutePath());
      }
    } else {
      // Read script
      _script = paramAccessor.getRequiredParameter(PROPERTY_SCRIPT);
    }

    // Create engine
    _engines = new ScriptEngineManager();
    _engine = _engines.getEngineByMimeType(paramAccessor.getParameter(PROPERTY_TYPE, "text/javascript"));
    // Check if engine supports multiple threads
    if (_engine.getFactory().getParameter("THREADING") == null) {
      _engine = null;
    } else {
      _engine.put("javax.script.filename", scriptFile != null ? scriptFile : "<inline script>");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _configuration);
    final ResultIDs resultCollector = new ResultIDs(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);

    if (recordIds != null) {
      // Create engine, if it does not support multiple threads
      ScriptEngine engine = _engine;
      if (engine == null) {
        engine = _engines.getEngineByMimeType(paramAccessor.getParameter(PROPERTY_TYPE, "text/javascript"));
      }

      // Check script file for modification
      checkScriptModification();

      // Execute script for every record
      for (final String id : recordIds) {
        try {
          paramAccessor.setCurrentRecord(id);
          resultCollector._excludeCurrentRecord = false;
          final Bindings bindings = engine.createBindings();
          final AnyMap metadata = blackboard.getMetadata(id);
          bindings.put(BLACKBOARD_VARIABLE, blackboard);
          bindings.put(ID_VARIABLE, id);
          bindings.put(METADATA_VARIABLE, metadata);
          bindings.put(RESULTS_VARIABLE, resultCollector);
          bindings.put(PARAMETER_ACCESSOR_VARIABLE, paramAccessor);
          final Object result = engine.eval(_script, bindings);
          // Add record only to the result, if it is accepted by the script (by default, it is accepted)
          if (!resultCollector._excludeCurrentRecord) {
            final String resultAttribute = paramAccessor.getParameter(PROPERTY_RESULT_ATTRIBUTE, null);
            if (resultAttribute != null) {
              if (result == null) {
                metadata.remove(resultAttribute);
              } else {
                metadata.put(resultAttribute, convertObject(result));
              }
            }
            resultCollector.addResult(id);
          }
        } catch (final Exception e) {
          resultCollector.addFailedResult(id, e);
        }
      }
    }
    return resultCollector.getResultIds();
  }

  /**
   * Check the script file for modification.
   */
  private void checkScriptModification() throws ProcessingException {
    if (_scriptFile != null && (_script == null || _fileLastModified != _scriptFile.lastModified())) {
      if (_script == null) {
        _script = "";
      }
      _fileLastModified = _scriptFile.lastModified();
      try {
        final FileInputStream in = new FileInputStream(_scriptFile);
        try {
          _script = IOUtils.toString(in);
        } finally {
          in.close();
        }
      } catch (final IOException e) {
        throw new ProcessingException("Could not read script file: " + _scriptFile.getAbsolutePath(), e);
      }
    }
  }

  /**
   * Converts an arbitrary native object to an {@link Any}.
   * 
   * @param object
   *          the object to convert
   * @return the "any" object
   */
  private static Any convertObject(final Object object) {
    if (object == null) {
      return null;
    } else if (object instanceof Any) {
      return (Any) object;
    } else if (object instanceof Map<?, ?>) {
      return convertMap((Map<?, ?>) object);
    } else if (object instanceof Iterable<?>) {
      return convertCollection((Iterable<?>) object);
    } else if (object instanceof Object[]) {
      return convertCollection(Arrays.asList((Object[]) object));
    }
    try {
      return DataFactory.DEFAULT.autoConvertValue(object);
    } catch (final InvalidValueTypeException e) {
      // Use the string representation
      return DataFactory.DEFAULT.createStringValue(object.toString());
    }
  }

  /**
   * Converts a collection (or similar) to a {@link AnySeq}.
   * 
   * @param collection
   *          the collection
   * @return the "any sequence"
   */
  private static Any convertCollection(final Iterable<?> collection) {
    final AnySeq result = DataFactory.DEFAULT.createAnySeq();
    for (final Object o : collection) {
      if (o != null) {
        result.add(convertObject(o));
      }
    }
    return result;
  }

  /**
   * Converts a map to a {@link AnyMap}.
   * 
   * @param map
   *          the map
   * @return the "any map"
   */
  private static AnyMap convertMap(final Map<?, ?> map) {
    final AnyMap result = DataFactory.DEFAULT.createAnyMap();
    for (final Entry<?, ?> entry : map.entrySet()) {
      if (entry.getKey() != null && entry.getValue() != null) {
        result.put(entry.getKey().toString(), convertObject(entry.getValue()));
      }
    }
    return result;
  }

  /**
   * An enhanced result collector that is able to drop the current record.
   */
  public static final class ResultIDs extends ResultCollector {
    /** Indicatesto remove the current processed record from the list of result ids. */
    private boolean _excludeCurrentRecord;

    /**
     * Creates a new instance of ResultIDs.
     * 
     * @param paramAccessor
     *          accessor of the hosting config, needed to access failOnError parameter
     * @param log
     *          the log for which to log the error results
     * @param dropRecordOnError
     *          indicates to drop records if the fail
     */
    public ResultIDs(final ParameterAccessor paramAccessor, final Log log, final boolean dropRecordOnError) {
      super(paramAccessor, log, dropRecordOnError);
    }

    /**
     * Call to remove the current record id from the list of result ids.
     */
    public void excludeCurrentRecord() {
      _excludeCurrentRecord = true;
    }

  }

}
