/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Marco Strack (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.scripting.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.scripting.ScriptNotFoundException;
import org.eclipse.smila.scripting.ScriptingEngine;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.eclipse.smila.utils.MaybeRecoverableException;

/**
 * Handles ReST calls for executing and displaying infos about a single script function.
 */
public class ScriptExecutionHandler extends JsonRequestHandler {

  /** The reference to the scripting service. */
  private ScriptingEngine _scriptingEngine;

  /**
   * Set GET instead of POST as basic method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return isGetRequest(method) || isPostRequest(method);
  }

  /**
   * accept GET requests.
   */
  protected boolean isGetRequest(final String method) {
    return "GET".equals(method);
  }

  /**
   * accept POST requests.
   */
  protected boolean isPostRequest(final String method) {
    return "POST".equals(method);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord,
    final HttpExchange exchange) throws Exception {
    //AnyMap result = FACTORY.createAnyMap();
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() == 1) {
      final String scriptName = uriParts.get(0);
      if (isGetRequest(method)) {
        final AnySeq itemsSeq = _scriptingEngine.listScripts();
        for (Any singleItem : itemsSeq) {
          final String itemName = singleItem.asMap().getStringValue("name");
          if (itemName.equals(scriptName)) {
            singleItem.asMap().put("url", URLCreator.create(getRequestHost(exchange), requestUri).toString());
            return singleItem;
          }
        }
        throw new ScriptNotFoundException("Unable to find " + scriptName);
      } else { //POST
        return _scriptingEngine.callScript(scriptName, inputRecord);
      }
    }
    return null;
  }

  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof ScriptNotFoundException) {
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof MaybeRecoverableException) {
      if (((MaybeRecoverableException) ex).isRecoverable()) {
        return HttpStatus.SERVICE_UNAVAILABLE;
      }
    }
    if (ex instanceof ScriptingEngineException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /** set OSGI service. */
  public void setScriptingEngine(final ScriptingEngine scriptingEngine) {
    _scriptingEngine = scriptingEngine;
  }

  /** unset OSGI service. */
  public void unsetScriptingEngine(final ScriptingEngine scriptingEngine) {
    if (_scriptingEngine == scriptingEngine) {
      _scriptingEngine = null;
    }
  }
}
