/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Iterator;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Attachment;
import org.eclipse.smila.datamodel.Record;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.ScriptRuntime;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.Wrapper;

/**
 * Provides access to Record attachments as a native Javascript object.
 */
public class AttachmentWrapper extends ObjectWrapperBase {
  protected static final Charset STRING_AS_ATTACHMENT_CHARSET = Charset.forName("UTF-8");

  private final Record _record;

  /** Creates wrapper instance. */
  public AttachmentWrapper(final Record record, final Scriptable parentScope) {
    super(parentScope);
    _record = record;
  }

  @Override
  public Record unwrap() {
    return _record;
  }

  @Override
  public String getClassName() {
    return "AttachmentWrapper";
  }

  /**
   * @return attachment from contained record as {@link Attachment}.
   */
  @Override
  public Object get(final String name, final Scriptable start) {
    return _record.getAttachment(name);
  }

  @Override
  public Object get(final int index, final Scriptable start) {
    throw Context.reportRuntimeError("Number keys are not allowed for Attachments.");
  }

  @Override
  public boolean has(final String name, final Scriptable start) {
    return _record.hasAttachment(name);
  }

  @Override
  public boolean has(final int index, final Scriptable start) {
    throw Context.reportRuntimeError("Number keys are not allowed for Attachments.");
  }

  /**
   * Supported value types are:
   * <ul>
   * <li>byte[]</li>: is used as the attachment value immediately.
   * <li>{@link CharSequence}</li>: is converted to a String and then to a byte[] using UTF-8 charset.
   * <li>{@link Attachment}</li>: if the name of the attachment matches the given name, the attachment object is added
   * to the wrapped record itself. Else the method tries to get the attachment content as a byte[] and adds it to the
   * wrapped record. If the given attachment does not return the attachment content as byte[], an EvaluatorException
   * will be thrown.
   * <li>{@link InputStream}</li>: The method tries to read the stream into an byte[] and sets this as the attachment.
   * If reading fails, an EvaluatorException will be thrown. The given stream is closed afterwards in any case.
   * </ul>
   *
   * The value can also be a {@link Wrapper}. In this case unwrapping must produce one of the above types.
   *
   * @param value
   *          an object to create an attachment value from.
   */
  @Override
  public void put(final String name, final Scriptable start, Object value) {
    if (value instanceof Wrapper) {
      value = ((Wrapper) value).unwrap();
    }

    if (value instanceof byte[]) {
      _record.setAttachment(name, (byte[]) value);

    } else if (value instanceof CharSequence) {
      _record.setAttachment(name, value.toString().getBytes(STRING_AS_ATTACHMENT_CHARSET));

    } else if (value instanceof Attachment) {
      final Attachment attachment = (Attachment) value;
      if (!name.equals(attachment.getName())) {
        final byte[] attachmentValue = attachment.getAsBytes();
        if (attachmentValue != null) {
          _record.setAttachment(name, attachmentValue);
        } else {
          throw Context.reportRuntimeError("Cannot copy attachment content from attachment '"
            + attachment.getName() + "' to new attachment '" + name + "'");
        }
      } else {
        _record.setAttachment(attachment);
      }

    } else if (value instanceof InputStream) {
      final InputStream attachmentStream = (InputStream) value;
      try {
        final byte[] attachment = IOUtils.toByteArray(attachmentStream);
        _record.setAttachment(name, attachment);
      } catch (final IOException ex) {
        throw Context.reportRuntimeError("Error reading InputStream to attachment '" + name + "': " + ex);
      } finally {
        IOUtils.closeQuietly(attachmentStream);
      }

    } else {
      throw Context.reportRuntimeError("Invalid attachment type " + value.getClass());
    }
  }

  @Override
  public void put(final int index, final Scriptable start, final Object value) {
    throw Context.reportRuntimeError("Number keys are not allowed for Attachments.");
  }

  @Override
  public void delete(final String name) {
    _record.removeAttachment(name);
  }

  @Override
  public void delete(final int index) {
    throw Context.reportRuntimeError("Number keys are not allowed for Attachments.");
  }

  @Override
  public Object[] getIds() {
    final String[] ids = new String[_record.attachmentSize()];
    final Iterator<String> names = _record.getAttachmentNames();
    int i = 0;
    while (names.hasNext()) {
      ids[i++] = names.next();
    }
    return ids;
  }

  @Override
  public Object getDefaultValue(final Class<?> hint) {
    if (hint == null || hint == ScriptRuntime.StringClass) {
      return Arrays.toString(getIds());
    }
    return super.getDefaultValue(hint);
  }
}
