/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Andreas Weber (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.PipeletTracker;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.mozilla.javascript.Scriptable;

/**
 * An instance of this class is registered in the scripting context for being able to create and access pipelets in the
 * JavaScript scripts.
 */
public class PipeletBuilder {
  private final PipeletTracker _pipeletTracker;

  private final BlackboardFactory _blackboardFactory;

  private final Scriptable _baseScope;

  /** Creates instance. */
  public PipeletBuilder(final PipeletTracker pipeletTracker, final BlackboardFactory blackboardFactory,
    final Scriptable baseScope) {
    _pipeletTracker = pipeletTracker;
    _blackboardFactory = blackboardFactory;
    _baseScope = baseScope;
  }

  /**
   * Instantiate and wrap a pipelet.
   * 
   * @param className
   *          name of pipelet class.
   * @param configuration
   *          pipelet configuration. May be null, must be convertible (or unwrappable) to an AnyMap object.
   * @return
   * @throws ScriptingEngineException
   */
  public PipeletWrapper create(final String className, final Object configuration) throws ScriptingEngineException {
    final Class<? extends Pipelet> pipeletClass = _pipeletTracker.getRegisteredPipelets().get(className);
    if (pipeletClass == null) {
      throw new ScriptingEngineException("Unknown pipelet class " + className);
    }
    try {
      final Pipelet pipelet = pipeletClass.newInstance();
      configurePipelet(pipelet, configuration);
      return new PipeletWrapper(pipelet, _blackboardFactory, _baseScope);
    } catch (IllegalAccessException | InstantiationException e) {
      throw new ScriptingEngineException("Error while creating instance for pipelet " + className, e);
    }
  }

  /**
   * Try to convert configuration object to an AnyMap and configure the pipelet.
   * 
   * @param pipelet
   *          pipelet instance.
   * @param configuration
   *          configuration object.
   * @throws ScriptingEngineException
   *           could not convert a non-null or non-Undefined configuration object to an AnyMap, or it was an invalid
   *           configuration.
   */
  private void configurePipelet(final Pipelet pipelet, final Object configuration) throws ScriptingEngineException {
    if (configuration != null) {
      Any anyConfig;
      if (configuration instanceof Record) {
        anyConfig = ((Record) configuration).getMetadata();
      } else if (configuration instanceof RecordWrapper) {
        anyConfig = ((RecordWrapper) configuration).unwrap().getMetadata();
      } else {
        anyConfig = AnyWrapper.unwrapAny(configuration);
      }
      if (anyConfig != null) {
        if (anyConfig instanceof AnyMap) {
          try {
            pipelet.configure((AnyMap) anyConfig);
          } catch (final ProcessingException e) {
            throw new ScriptingEngineException(
              "Error while configuring instance for pipelet " + pipelet.getClass(), e);
          }
        } else {
          throw new ScriptingEngineException("Configuration must be a Map object.");
        }
      }
    }
  }

}
