/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license agreements. See the NOTICE
 * file distributed with this work for additional information regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the
 * License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package org.apache.ode.bpel.memdao;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.TransactionManager;
import javax.xml.namespace.QName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ode.bpel.common.BpelEventFilter;
import org.apache.ode.bpel.common.Filter;
import org.apache.ode.bpel.common.InstanceFilter;
import org.apache.ode.bpel.common.ProcessFilter;
import org.apache.ode.bpel.dao.BpelDAOConnection;
import org.apache.ode.bpel.dao.MessageExchangeDAO;
import org.apache.ode.bpel.dao.ProcessDAO;
import org.apache.ode.bpel.dao.ProcessInstanceDAO;
import org.apache.ode.bpel.dao.ScopeDAO;
import org.apache.ode.bpel.evt.BpelEvent;
import org.apache.ode.utils.ISO8601DateParser;
import org.apache.ode.utils.stl.CollectionsX;
import org.apache.ode.utils.stl.UnaryFunction;

/**
 * A very simple, in-memory implementation of the {@link BpelDAOConnection} interface.
 */
class BpelDAOConnectionImpl implements BpelDAOConnection {
  private static final Log __log = LogFactory.getLog(BpelDAOConnectionImpl.class);

  public static long MIN_TIME_TO_LIVE = 2 * 60;

  private final TransactionManager _txm;

  private final Map<QName, ProcessDaoImpl> _store;

  private final List<BpelEvent> _events = new LinkedList<BpelEvent>();

  private final List<MessageExchangeDAOImpl> _mexList = new LinkedList<MessageExchangeDAOImpl>();

  private final Map<String, MessageExchangeDAOImpl> _mexStore = new HashMap<String, MessageExchangeDAOImpl>();

  private long _lastRemoval = 0;

  private final long _timeToLiveMillis;

  BpelDAOConnectionImpl(final Map<QName, ProcessDaoImpl> store, final TransactionManager txm,
    final int timeToLiveSecs) {
    _store = store;
    _txm = txm;
    // TTL = 2 * pipeline.timeout, but at least 2 minutes (original value ... we don't want to clean up more often.)
    _timeToLiveMillis = Math.max(MIN_TIME_TO_LIVE, 2 * timeToLiveSecs) * 1000;
  }

  @Override
  public ProcessDAO getProcess(final QName processId) {
    return _store.get(processId);
  }

  @Override
  public ProcessDAO createProcess(final QName pid, final QName type, final String guid, final long version) {
    final ProcessDaoImpl process = new ProcessDaoImpl(this, _store, pid, type, guid, version);
    _store.put(pid, process);
    return process;
  }

  @Override
  public ProcessInstanceDAO getInstance(final Long iid) {
    for (final ProcessDaoImpl proc : _store.values()) {
      final ProcessInstanceDAO instance = proc._instances.get(iid);
      if (instance != null) {
        return instance;
      }
    }
    return null;
  }

  @Override
  public Collection<ProcessInstanceDAO> instanceQuery(final InstanceFilter filter) {
    if (filter.getLimit() == 0) {
      return Collections.EMPTY_LIST;
    }
    final List<ProcessInstanceDAO> matched = new ArrayList<ProcessInstanceDAO>();
    // Selecting
    selectionCompleted: for (final ProcessDaoImpl proc : _store.values()) {
      boolean pmatch = true;
      if (filter.getNameFilter() != null
        && !equalsOrWildcardMatch(filter.getNameFilter(), proc.getProcessId().getLocalPart())) {
        pmatch = false;
      }
      if (filter.getNamespaceFilter() != null
        && !equalsOrWildcardMatch(filter.getNamespaceFilter(), proc.getProcessId().getNamespaceURI())) {
        pmatch = false;
      }

      if (pmatch) {
        for (final ProcessInstanceDAO inst : proc._instances.values()) {
          boolean match = true;

          if (filter.getStatusFilter() != null) {
            boolean statusMatch = false;
            for (final Short status : filter.convertFilterState()) {
              if (inst.getState() == status.byteValue()) {
                statusMatch = true;
              }
            }
            if (!statusMatch) {
              match = false;
            }
          }
          if (filter.getStartedDateFilter() != null
            && !dateMatch(filter.getStartedDateFilter(), inst.getCreateTime(), filter)) {
            match = false;
          }
          if (filter.getLastActiveDateFilter() != null
            && !dateMatch(filter.getLastActiveDateFilter(), inst.getLastActiveTime(), filter)) {
            match = false;
          }

          // if (filter.getPropertyValuesFilter() != null) {
          // for (Map.Entry propEntry : filter.getPropertyValuesFilter().entrySet()) {
          // boolean entryMatched = false;
          // for (ProcessPropertyDAO prop : proc.getProperties()) {
          // if (prop.getName().equals(propEntry.getKey())
          // && (propEntry.getValue().equals(prop.getMixedContent())
          // || propEntry.getValue().equals(prop.getSimpleContent()))) {
          // entryMatched = true;
          // }
          // }
          // if (!entryMatched) {
          // match = false;
          // }
          // }
          // }

          if (match) {
            matched.add(inst);
            if (matched.size() == filter.getLimit()) {
              break selectionCompleted;
            }
          }
        }
      }
    }
    // And ordering
    if (filter.getOrders() != null) {
      final List<String> orders = filter.getOrders();

      Collections.sort(matched, new Comparator<ProcessInstanceDAO>() {
        @Override
        public int compare(final ProcessInstanceDAO o1, final ProcessInstanceDAO o2) {
          for (final String orderKey : orders) {
            final int result = compareInstanceUsingKey(orderKey, o1, o2);
            if (result != 0) {
              return result;
            }
          }
          return 0;
        }
      });
    }

    return matched;
  }

  /**
   * Close this DAO connection.
   */
  @Override
  public void close() {
  }

  public Collection<ProcessDAO> processQuery(final ProcessFilter filter) {
    throw new UnsupportedOperationException("Can't query process configuration using a transient DAO.");
  }

  @Override
  public MessageExchangeDAO createMessageExchange(final String mexId, final char dir) {
    final MessageExchangeDAOImpl mex = new MessageExchangeDAOImpl(dir, mexId);
    mex.createTime = new Date();

    // FIXME: Why is this necessary? We should explicitly remove these thigs -mbs

    synchronized (_mexStore) {
      _mexStore.put(mexId, mex);
      _mexList.add(mex);
    }

    cleanupDeadWood();

    // Removing right away on rollback
    onRollback(new Runnable() {
      @Override
      public void run() {
        synchronized (_mexStore) {
          final MessageExchangeDAOImpl mexdao = _mexStore.remove(mexId);

          if (mexdao != null) {
            _mexList.remove(mexdao);
          }
        }
      }
    });

    return mex;
  }

  /**
   * Remove old message exchanges from the Mex store.
   * 
   */
  private void cleanupDeadWood() {
    final long now = System.currentTimeMillis();

    if (now > _lastRemoval + _timeToLiveMillis / 4) {
      _lastRemoval = now;

      synchronized (_mexStore) {
        final LinkedList<MessageExchangeDAOImpl> trash = new LinkedList<>();
        for (final MessageExchangeDAOImpl mexdao : _mexList) {
          final long createtime = mexdao._createTime.getTime();
          if (now - createtime > _timeToLiveMillis) {
            trash.add(mexdao);
          } else {
            break;
          }
        }

        _mexList.removeAll(trash);
        _mexStore.values().removeAll(trash);
      }
    }

  }

  @Override
  public MessageExchangeDAO getMessageExchange(final String mexid) {
    synchronized (_mexStore) {
      return _mexStore.get(mexid);
    }
  }

  @Override
  public void releaseMessageExchange(final String mexid) {
    synchronized (_mexStore) {
      final MessageExchangeDAO mexDao = _mexStore.remove(mexid);
      if (mexDao != null) {
        _mexList.remove(mexDao);
      }
    }
  }

  private int compareInstanceUsingKey(final String key, final ProcessInstanceDAO instanceDAO1,
    final ProcessInstanceDAO instanceDAO2) {
    String s1 = null;
    String s2 = null;
    boolean ascending = true;
    String orderKey = key;
    if (key.startsWith("+") || key.startsWith("-")) {
      orderKey = key.substring(1, key.length());
      if (key.startsWith("-")) {
        ascending = false;
      }
    }
    final ProcessDAO process1 = getProcess(instanceDAO1.getProcess().getProcessId());
    final ProcessDAO process2 = getProcess(instanceDAO2.getProcess().getProcessId());
    if ("pid".equals(orderKey)) {
      s1 = process1.getProcessId().toString();
      s2 = process2.getProcessId().toString();
    } else if ("name".equals(orderKey)) {
      s1 = process1.getProcessId().getLocalPart();
      s2 = process2.getProcessId().getLocalPart();
    } else if ("namespace".equals(orderKey)) {
      s1 = process1.getProcessId().getNamespaceURI();
      s2 = process2.getProcessId().getNamespaceURI();
    } else if ("version".equals(orderKey)) {
      s1 = "" + process1.getVersion();
      s2 = "" + process2.getVersion();
    } else if ("status".equals(orderKey)) {
      s1 = "" + instanceDAO1.getState();
      s2 = "" + instanceDAO2.getState();
    } else if ("started".equals(orderKey)) {
      s1 = ISO8601DateParser.format(instanceDAO1.getCreateTime());
      s2 = ISO8601DateParser.format(instanceDAO2.getCreateTime());
    } else if ("last-active".equals(orderKey)) {
      s1 = ISO8601DateParser.format(instanceDAO1.getLastActiveTime());
      s2 = ISO8601DateParser.format(instanceDAO2.getLastActiveTime());
    }
    if (ascending) {
      return s1.compareTo(s2);
    } else {
      return s2.compareTo(s1);
    }
  }

  private boolean equalsOrWildcardMatch(final String s1, final String s2) {
    if (s1 == null || s2 == null) {
      return false;
    }
    if (s1.equals(s2)) {
      return true;
    }
    if (s1.endsWith("*")) {
      if (s2.startsWith(s1.substring(0, s1.length() - 1))) {
        return true;
      }
    }
    if (s2.endsWith("*")) {
      if (s1.startsWith(s2.substring(0, s2.length() - 1))) {
        return true;
      }
    }
    return false;
  }

  public boolean dateMatch(final List<String> dateFilters, final Date instanceDate, final InstanceFilter filter) {
    boolean match = true;
    for (final String ddf : dateFilters) {
      final String isoDate = ISO8601DateParser.format(instanceDate);
      final String critDate = Filter.getDateWithoutOp(ddf);
      if (ddf.startsWith("=")) {
        if (!isoDate.startsWith(critDate)) {
          match = false;
        }
      } else if (ddf.startsWith("<=")) {
        if (!isoDate.startsWith(critDate) && isoDate.compareTo(critDate) > 0) {
          match = false;
        }
      } else if (ddf.startsWith(">=")) {
        if (!isoDate.startsWith(critDate) && isoDate.compareTo(critDate) < 0) {
          match = false;
        }
      } else if (ddf.startsWith("<")) {
        if (isoDate.compareTo(critDate) > 0) {
          match = false;
        }
      } else if (ddf.startsWith(">")) {
        if (isoDate.compareTo(critDate) < 0) {
          match = false;
        }
      }
    }
    return match;
  }

  @Override
  public ScopeDAO getScope(final Long siidl) {
    for (final ProcessDaoImpl process : _store.values()) {
      for (final ProcessInstanceDAO instance : process._instances.values()) {
        if (instance.getScope(siidl) != null) {
          return instance.getScope(siidl);
        }
      }
    }
    return null;
  }

  @Override
  public void insertBpelEvent(final BpelEvent event, final ProcessDAO processConfiguration,
    final ProcessInstanceDAO instance) {
    _events.add(event);
  }

  @Override
  public List<Date> bpelEventTimelineQuery(final InstanceFilter ifilter, final BpelEventFilter efilter) {
    // TODO : Provide more correct implementation:
    final ArrayList<Date> dates = new ArrayList<Date>();
    CollectionsX.transform(dates, _events, new UnaryFunction<BpelEvent, Date>() {
      @Override
      public Date apply(final BpelEvent x) {
        return x.getTimestamp();
      }
    });
    return dates;
  }

  @Override
  public List<BpelEvent> bpelEventQuery(final InstanceFilter ifilter, final BpelEventFilter efilter) {
    // TODO : Provide a more correct (filtering) implementation:
    return _events;
  }

  /**
   * @see org.apache.ode.bpel.dao.BpelDAOConnection#instanceQuery(String)
   */
  @Override
  public Collection<ProcessInstanceDAO> instanceQuery(final String expression) {
    // TODO
    throw new UnsupportedOperationException();
  }

  public void defer(final Runnable runnable) {
    try {
      _txm.getTransaction().registerSynchronization(new Synchronization() {
        @Override
        public void afterCompletion(final int status) {
        }

        @Override
        public void beforeCompletion() {
          runnable.run();
        }
      });

    } catch (final Exception e) {
      throw new RuntimeException(e);
    }
  }

  public void onRollback(final Runnable runnable) {
    try {
      _txm.getTransaction().registerSynchronization(new Synchronization() {
        @Override
        public void afterCompletion(final int status) {
          if (status != Status.STATUS_COMMITTED) {
            runnable.run();
          }
        }

        @Override
        public void beforeCompletion() {
        }
      });
    } catch (final Exception e) {
      throw new RuntimeException(e);
    }
  }

}
