/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyConvertException;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;

/**
 * The description of a task execution result.
 */
public class ResultDescription {
  /** Key for status. */
  public static final String KEY_STATUS = "status";

  /** Key for errorCode. */
  public static final String KEY_ERROR_CODE = "errorCode";

  /** Key for counters. */
  public static final String KEY_COUNTERS = "counters";

  /** Key for errorMessage. */
  public static final String KEY_ERROR_MESSAGE = "errorMessage";

  /** The completion status of the task. */
  private final TaskCompletionStatus _status;

  /** An optional error code. */
  private final String _errorCode;

  /** An optional error message. */
  private final String _errorMessage;

  /** Counters for statistical purpose. */
  private final Map<String, Number> _counters;

  /**
   * Constructs a new ResultDescription.
   * 
   * @param status
   *          The completion status.
   * @param errorCode
   *          An error code if an error occurred, null (or a success code) if not.
   * @param errorMessage
   *          An error message if an error occurred, null if not.
   * @param counters
   *          Counters for statistical purposes.
   */
  public ResultDescription(final TaskCompletionStatus status, final String errorCode, final String errorMessage,
    final Map<String, Number> counters) {
    _status = status;
    _errorCode = errorCode;
    _errorMessage = errorMessage;
    if (counters != null) {
      _counters = counters;
    } else {
      _counters = new HashMap<>();
    }
  }

  /**
   * Parse a new ResultDescription from an Any object.
   * 
   * @param resultDescriptionAny
   *          see class comment for format of Any object
   * @return a ResultDescription constructed from the Any object
   * @throws TaskmanagerException
   * @throws AnyConvertException
   *           error converting object.
   */
  public static ResultDescription fromAny(final AnyMap resultDescriptionAny) throws AnyConvertException {
    try {
      final TaskCompletionStatus status =
        TaskCompletionStatus.valueOf(resultDescriptionAny.getStringValue(KEY_STATUS));
      final String errorCode;
      final String errorMessage;
      final Map<String, Number> counters;
      if (resultDescriptionAny.get(KEY_ERROR_CODE) != null) {
        errorCode = resultDescriptionAny.getStringValue(KEY_ERROR_CODE);
      } else {
        errorCode = null;
      }
      if (resultDescriptionAny.get(KEY_ERROR_MESSAGE) != null) {
        errorMessage = resultDescriptionAny.getStringValue(KEY_ERROR_MESSAGE);
      } else {
        errorMessage = null;
      }
      if (resultDescriptionAny.get(KEY_COUNTERS) != null) {
        counters = new LinkedHashMap<String, Number>();
        final AnyMap countersAny = resultDescriptionAny.getMap(KEY_COUNTERS);
        for (final Entry<String, Any> entry : countersAny.entrySet()) {
          final String key = entry.getKey();
          final Any anyValue = entry.getValue();
          final Number number;
          if (anyValue.isDouble()) {
            number = ((Value) anyValue).asDouble();
          } else if (anyValue.isLong()) {
            number = ((Value) anyValue).asLong();
          } else {
            throw new AnyConvertException("Invalid Any type '" + anyValue.getValueType() + "' for counter '" + key
              + "'.");
          }
          counters.put(key, number);
        }
      } else {
        counters = null;
      }

      return new ResultDescription(status, errorCode, errorMessage, counters);
    } catch (final Exception ex) {
      throw new AnyConvertException("Error parsing ResultDescription from Any object", ex);
    }
  }

  /**
   * @return the status
   */
  public TaskCompletionStatus getStatus() {
    return _status;
  }

  /**
   * @return the errorCode
   */
  public String getErrorCode() {
    return _errorCode;
  }

  /**
   * @return the errorMessage
   */
  public String getErrorMessage() {
    return _errorMessage;
  }

  /**
   * @return the counters
   */
  public Map<String, Number> getCounters() {
    return _counters;
  }

  /**
   * Returns the Any representation of the result description.
   * 
   * @return the Any representation of the result description
   * @throws Exception
   *           an exception occurred during the conversion to Any.
   */
  public AnyMap toAny() throws Exception {
    final AnyMap result = DataFactory.DEFAULT.createAnyMap();
    result.put(KEY_STATUS, _status.name());
    if (_errorCode != null) {
      result.put(KEY_ERROR_CODE, _errorCode);
    }
    if (_errorMessage != null) {
      result.put(KEY_ERROR_MESSAGE, _errorMessage);
    }
    if (_counters != null) {
      final AnyMap anyCounters = DataFactory.DEFAULT.createAnyMap();
      for (final Entry<String, Number> entry : _counters.entrySet()) {
        final Number number = entry.getValue();
        anyCounters.put(entry.getKey(), number);
      }
      result.put(KEY_COUNTERS, anyCounters);
    }
    return result;
  }
}
