/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.importing.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.taskworker.TaskContext;

final public class PropertyNameMapper {

  final AnyMap _mapping;

  final boolean _normalize;

  public PropertyNameMapper(final AnyMap mapping) {
    this(mapping, false);
  }

  public PropertyNameMapper(final AnyMap mapping, final boolean normalize) {
    if (mapping == null) {
      throw new NullPointerException("Parameter mapping is null");
    }
    if (mapping.size() == 0) {
      throw new IllegalArgumentException("Parameter mapping is empty");
    }
    _normalize = normalize;
    if (normalize) {
      _mapping = mapping.getFactory().createAnyMap();
      for (final Entry<String, Any> entry : mapping.entrySet()) {
        _mapping.put(entry.getKey().toUpperCase(Locale.ENGLISH), entry.getValue());
      }
    } else {
      _mapping = mapping;
    }
  }

  public static PropertyNameMapper createFrom(final TaskContext taskContext) {
    if (taskContext == null) {
      throw new NullPointerException("Parameter taskContext is null");
    }
    final AnyMap taskParams = taskContext.getTaskParameters();
    return createFrom(taskParams);
  }

  public static PropertyNameMapper createFrom(final AnyMap taskParams) {
    return createFrom(taskParams, false);
  }

  public static PropertyNameMapper createFrom(final AnyMap taskParams, final boolean normalize) {
    final AnyMap mapping = taskParams.getMap(ImportingConstants.TASK_PARAM_MAPPING);
    if (mapping == null) {
      throw new IllegalArgumentException("Parameter '" + ImportingConstants.TASK_PARAM_MAPPING + " is null");
    }
    if (mapping.size() == 0) {
      throw new IllegalArgumentException("Parameter '" + ImportingConstants.TASK_PARAM_MAPPING + " is empty");
    }
    return new PropertyNameMapper(mapping, normalize);
  }

  public List<String> get(String key) {
    final List<String> values = new ArrayList<String>();
    if (_normalize) {
      key = key.toUpperCase(Locale.ENGLISH);
    }

    if (!_mapping.containsKey(key)) {
      throw new RuntimeException("No mapping for key '" + key + "' defined");
    }
    for (final Any anyVal : _mapping.get(key)) {
      values.add(anyVal.asValue().asString());
    }
    return values;
  }

  public boolean containsMapping(String key) {
    if (_normalize) {
      key = key.toUpperCase(Locale.ENGLISH);
    }
    return _mapping.containsKey(key);
  }

  public void mapNames(final Record record, final Set<String> crawlerPropertyNames) {
    final AnyMap metadata = record.getMetadata();
    for (final Entry<String, Any> entry : _mapping.entrySet()) {
      final String key = entry.getKey();
      boolean removeKey = true;
      for (final Any mappedValue : entry.getValue()) {
        final String mappedKey = mappedValue.asValue().asString();
        if (!mappedKey.equals(key)) {
          if (metadata.containsKey(key)) {
            metadata.put(mappedKey, metadata.get(key));
          } else if (record.hasAttachment(key)) {
            record.setAttachment(mappedKey, record.getAttachmentAsBytes(key)); // TODO: better to rename attachment key
                                                                               // internally ???
          }
        } else {
          // do not remove original key, it is (also) mapped to itself.
          removeKey = false;
        }
      }
      if (removeKey) {
        if (metadata.containsKey(key)) {
          metadata.remove(key);
        } else if (record.hasAttachment(key)) {
          record.removeAttachment(key);
        }
      }
    }

    // remove all properties for which no mapping is configured
    if (crawlerPropertyNames != null) {
      for (final String name : crawlerPropertyNames) {
        if (!containsMapping(name)) {
          metadata.remove(name);
          record.removeAttachment(name);
        }
      }
    }
  }

  public AnySeq getMappingKeys() {
    final AnySeq keys = DataFactory.DEFAULT.createAnySeq();
    for (final String key : _mapping.keySet()) {
      keys.add(key);
    }
    return keys;
  }
}
