/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.definitions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Locale;

import org.eclipse.smila.common.exceptions.InvalidDefinitionException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;

/** Available Job Run Modes. */
public enum JobRunMode {
  /** standard mode (default). */
  STANDARD("standard"),
  /** run once mode. */
  RUNONCE("runOnce");

  /** list of all job run modes, in default priority order. */
  public static final List<JobRunMode> ALL_MODES = Arrays.asList(STANDARD, RUNONCE);

  /** job mode. */
  private final String _mode;

  /**
   * @param n
   *          mode name
   */
  JobRunMode(final String n) {
    _mode = n;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return _mode;
  }

  /**
   * convert sequence of mode values to list of JobRunModes.
   * 
   * @param modesAny
   *          may be null.
   * @return null if input was null or empty, else list of corresponding JobRunModes.
   * @throws InvalidDefinitionException
   *           if input contains non-value anys or invalid values.
   */

  public static List<JobRunMode> parseModes(final AnySeq modesAny) throws InvalidDefinitionException {
    if (modesAny == null || modesAny.isEmpty()) {
      return null;
    } else {
      final List<JobRunMode> modes = new ArrayList<JobRunMode>(modesAny.size());
      for (final Any any : modesAny) {
        if (any.isValue()) {
          final String modeName = any.asValue().asString();
          try {
            modes.add(JobRunMode.valueOf(modeName.toUpperCase(Locale.ENGLISH)));
          } catch (final Exception ex) {
            throw new InvalidDefinitionException("Modes list contains invalid value '" + modeName + "'");
          }
        } else {
          throw new InvalidDefinitionException("Modes list contains non-value objects.");
        }
      }
      return modes;
    }
  }

  /**
   * convert modes to corresponding sequence of string values.
   * 
   * @param modes
   *          must not be null.
   * @return corresponding sequence.
   */
  public static AnySeq toAny(final Collection<JobRunMode> modes) {
    final AnySeq modesAny = DataFactory.DEFAULT.createAnySeq();
    for (final JobRunMode mode : modes) {
      modesAny.add(mode.toString());
    }
    return modesAny;
  }
}
