/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.TreeMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.common.definitions.ParameterDefinition;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.JobRun;
import org.eclipse.smila.jobmanager.JobRunDataProvider;
import org.eclipse.smila.jobmanager.JobRunEngine;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;
import org.eclipse.smila.jobmanager.definitions.JobRunDefinitions;
import org.eclipse.smila.jobmanager.definitions.JobRunMode;
import org.eclipse.smila.jobmanager.definitions.WorkerDefinition;
import org.eclipse.smila.jobmanager.definitions.WorkflowAction;
import org.eclipse.smila.jobmanager.events.JobListener;
import org.eclipse.smila.jobmanager.events.PrepareToFinishEvent;
import org.eclipse.smila.jobmanager.exceptions.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.exceptions.IllegalJobStateException;
import org.eclipse.smila.jobmanager.exceptions.JobDependencyException;
import org.eclipse.smila.jobmanager.exceptions.JobManagerException;
import org.eclipse.smila.jobmanager.exceptions.JobRunModeNotAllowedException;
import org.eclipse.smila.jobmanager.exceptions.PersistenceException;
import org.eclipse.smila.jobmanager.persistence.JobRunListener;
import org.eclipse.smila.jobmanager.persistence.PermanentStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorageException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerationUtil;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerator;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorProvider;
import org.eclipse.smila.objectstore.NoSuchStoreException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;

/**
 * Implements {@link JobRunEngine}.
 */
public class JobRunEngineImpl implements JobRunEngine, JobRunListener {

  /** simple date format used for creating readable job id. */
  private static final int JOB_RUN_ID_SUFFIX_MAX = 1000;

  /** random element to make job ids unique. */
  private static final Random RANDOM = new Random(System.nanoTime());

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** service reference to run storage. */
  private RunStorage _runStorage;

  /** service reference to permanent storage. */
  private PermanentStorage _permStorage;

  /** wrapper for definition storage adding access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /** service reference to task manager. */
  private TaskManager _taskManager;

  /** service reference to task generator provider used to select a task generator. */
  private TaskGeneratorProvider _taskGeneratorProvider;

  /** service reference to object store. */
  private ObjectStoreService _objectStore;

  /** service reference to job data provider. */
  private JobRunDataProvider _dataProvider;

  /** the map with the job runs. job name as key, job run as value */
  private final Map<String, JobRun> _jobRuns = new HashMap<String, JobRun>();

  /** Listener for job events. */
  private final CopyOnWriteArrayList<JobListener> _jobListeners = new CopyOnWriteArrayList<JobListener>();

  /** {@inheritDoc} */
  @Override
  public String startJob(final String jobName) throws JobManagerException {
    return startJob(jobName, null);
  }

  /** {@inheritDoc} */
  @Override
  public String startJob(final String jobName, final JobRunMode requestedJobRunMode) throws JobManagerException {
    _log.info("start called for job '" + jobName + "', jobRunMode '" + requestedJobRunMode + "'");
    String jobId = null;
    JobState currentState = null;
    try {
      if (!_defPersistence.hasJob(jobName)) {
        throw new IllegalArgumentException("No definition for job '" + jobName + "' exists.");
      }
      // check if another job run of this job is already (or still) active.
      currentState = _runStorage.getJobState(jobName);
      if (currentState != null) {
        throw new IllegalStateException("Job '" + jobName + "' is in state '" + currentState + "'.");
      }
      final JobRunDefinitions jobRunDefs = _dataProvider.getJobRunDefinitions(jobName);
      final JobRunMode jobRunMode = checkJobRunMode(requestedJobRunMode, jobRunDefs);
      checkJobDependenciesForStarting(jobRunDefs);
      jobId = createJobId();
      _runStorage.startJobRun(jobName, jobId, jobRunMode, jobRunDefs);
      _runStorage.registerJobRunListener(this, jobName); // to avoid memory leaks in cluster
      // create job run and store it in map
      final JobRun jobRun = ensureJobRun(jobName, jobId);
      checkBuckets(jobRun);
      addJobTriggers(jobRun);
      if (_runStorage.setJobState(jobName, jobId, JobState.PREPARING, JobState.RUNNING)) {
        if (JobRunMode.RUNONCE == jobRunMode) {
          executeRunOnceJob(jobRun);
        }
        _log.info("started job run '" + jobId + "' for job '" + jobName + "'");
        return jobId;
      } else {
        throw new JobManagerException("Error while starting job '" + jobName + "' with jobId '" + jobId + "'. ");
      }
    } catch (final Exception e) {
      throw newJobStartFailure(jobName, jobId, currentState, e);
    }
  }

  /** clean up state and create appropriate exception after failed job run start. */
  private JobManagerException newJobStartFailure(final String jobName, final String jobId, JobState currentState,
    final Exception e) throws JobManagerException {
    final String messagePrefix = "Could not start job '" + jobName + "': ";
    if (jobId != null) {
      currentState = _runStorage.getJobState(jobName);
      if (currentState == null) {
        // job run could have been removed in between, e.g. by a cancel, so have a look in persistent storage
        AnyMap jobData = null;
        try {
          jobData = _dataProvider.getJobRunData(jobName, jobId, false);
        } catch (final ConfigNotFoundException ce) {
          ; // ignore
        }
        if (jobData != null && jobData.getStringValue(JobManagerConstants.DATA_JOB_STATE) != null) {
          currentState = JobState.valueOf(jobData.getStringValue(JobManagerConstants.DATA_JOB_STATE));
        }
      }
    }
    if (currentState == JobState.CANCELED || currentState == JobState.CANCELLING) {
      return new JobManagerException(messagePrefix + "Job run was canceled while starting.", e);
    }
    cleanupFailedJobRun(jobName, jobId);
    return new JobManagerException(messagePrefix + e.getMessage(), e);
  }

  /**
   * determine default job run mode from job or workflow definition, if requestRunMode is null. Else check if requested
   * mode is allowed by job or workflow definition.
   * 
   * @throws JobManagerException
   *           if requested mode is not allowed by definitions.
   */
  private JobRunMode checkJobRunMode(final JobRunMode requestedRunMode, final JobRunDefinitions jobRunDefs)
    throws JobManagerException {
    if (requestedRunMode == null) {
      JobRunMode defaultRunMode = jobRunDefs.getJobDefinition().getDefaultJobRunMode();
      if (defaultRunMode == null) {
        defaultRunMode = jobRunDefs.getWorkflowDefinition().getDefaultJobRunMode();
      }
      if (defaultRunMode == null) {
        return JobRunMode.STANDARD;
      }
      return defaultRunMode;
    } else {
      List<JobRunMode> allowedRunModes = jobRunDefs.getJobDefinition().getJobRunModes();
      if (allowedRunModes == null) {
        allowedRunModes = jobRunDefs.getWorkflowDefinition().getJobRunModes();
      }
      if (allowedRunModes == null || allowedRunModes.contains(requestedRunMode)) {
        return requestedRunMode;
      }
      throw new JobRunModeNotAllowedException("Job '" + jobRunDefs.getJobDefinition().getName()
        + "' or its workflow does not allow run mode '" + requestedRunMode + "'.");
    }
  }

  /** {@inheritDoc} */
  @Override
  public void finishJob(final String jobName, final String jobRunId) throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("finish called for job '" + jobName + "', run '" + jobRunId + "'");
    }
    final JobState state = _runStorage.getJobState(jobName, jobRunId);
    if (state == JobState.RUNNING) {
      // check if no dependent job is running.
      checkNoDependentJobIsRunning(jobName, jobRunId);
      // notify listeners...
      for (final JobListener jobListener : _jobListeners) {
        try {
          jobListener.processJobEvent(new PrepareToFinishEvent(jobName, jobRunId));
        } catch (final Throwable t) {
          _log.error("JobListener failed to process PrepareToFinishEvent for job '" + jobName + "', rob run id '"
            + jobRunId + "'.", t);
        }
      }
      if (_runStorage.finishJobRun(jobName, jobRunId)) {
        removeJobTriggers(ensureJobRun(jobName, jobRunId));
        checkAndHandleJobRunCompleted(jobName, jobRunId);
      }
    } else if (state == null) {
      throw newExceptionNoCurrentJobRun(jobName, jobRunId);
    } else {
      // found in RunStorage but not RUNNING
      throw new IllegalJobStateException("Job run '" + jobRunId + "' of job '" + jobName
        + "' couldn't be finished, because it's in state '" + state + "'.");
    }
  }

  /** {@inheritDoc} */
  @Override
  public void cancelJob(final String jobName, final String jobRunId) throws JobManagerException {
    final int maxTries = 3;
    int tries = 0;
    boolean success = false;
    JobState state = null;
    // loop to handle meantime job run state changes
    // (loop can only happen for state changes PREPRARING -> RUNNING -> FINISHING)
    do {
      tries++;
      state = _runStorage.getJobState(jobName, jobRunId);
      checkJobRunStateForJobRunCancelling(jobName, jobRunId, state);
      // set job run state on CANCELLING (only successful if former state is still valid)
      success = _runStorage.setJobState(jobName, jobRunId, state, JobState.CANCELLING);
    } while (!success && tries <= maxTries);

    try {
      // remove tasks in taskmanager
      removeTasksQuietly(jobName, jobRunId);

      // remove transient data for canceled workflow runs before canceling the workflow runs in run storage,
      // otherwise transient data won't be deleted, because info about what to delete is lost.
      final List<String> workflowRunIds = _runStorage.getWorkflowRuns(jobName, jobRunId);
      for (final String workflowRun : workflowRunIds) {
        deleteTransientBulks(jobName, jobRunId, workflowRun);
      }

      // cancel workflow runs and tasks in run storage
      _runStorage.cancelJobRun(jobName, jobRunId, workflowRunIds);

      // persist job run data in objectstore, delete job run data in ZK, set job run state on CANCELED
      completeJobRun(jobName, jobRunId, JobState.CANCELED);
    } catch (final Exception e) {
      throw new JobManagerException("Error while canceling job run '" + jobRunId + "' of job '" + jobName + "'", e);
    }
  }

  /** check if current job run state allows job run cancelling. */
  private void checkJobRunStateForJobRunCancelling(final String jobName, final String jobRunId, final JobState state)
    throws JobManagerException {
    if (state == null) {
      throw newExceptionNoCurrentJobRun(jobName, jobRunId);
    }
    if (state == JobState.CANCELLING || state == JobState.CANCELED) {
      throw new IllegalJobStateException("Job run '" + jobRunId + "' of job '" + jobName
        + "' couldn't be canceled, because someone else canceled it before.");
    }
    if (state == JobState.SUCCEEDED || state == JobState.FAILED || state == JobState.CLEANINGUP) {
      throw new IllegalJobStateException("Job run '" + jobRunId + "' of job '" + jobName
        + "' couldn't be canceled, because it's in state '" + state + "'.");
    }
  }

  /** check if current job run state allows workflow run cancelling. */
  private void checkJobRunStateForWorkflowRunCancelling(final String jobName, final String jobRunId,
    final String workflowRunId, final JobState state) throws JobManagerException {
    if (state == null) {
      throw newExceptionNoCurrentJobRun(jobName, jobRunId);
    }
    if (state != JobState.RUNNING && state != JobState.FINISHING) {
      throw new IllegalJobStateException("Workflow run '" + workflowRunId + "' for job run '" + jobRunId
        + "' of job '" + jobName + "' is only allowed in job run state RUNNING or FINISHING, was: " + state);
    }
  }

  /** {@inheritDoc} */
  @Override
  public JobRun ensureJobRun(final String jobName, final String jobRunId) throws JobManagerException {
    JobRun jobRun = _jobRuns.get(jobName);
    if (jobRun == null || !jobRunId.equals(jobRun.getJobRunId())) {
      try {
        jobRun = new JobRunImpl(jobRunId, jobName, _runStorage, _defPersistence);
      } catch (final Exception e) {
        throw new JobManagerException("Error during creation of job run: ", e);
      }
      _jobRuns.put(jobName, jobRun);
    }
    return jobRun;
  }

  /** {@inheritDoc} */
  @Override
  public void deleteTransientBulks(final String jobName, final String jobRunId, final String workflowRunId) {
    try {
      final Collection<String> transientBulks = _runStorage.getTransientBulks(jobName, jobRunId, workflowRunId);
      for (final String transientBulkId : transientBulks) {
        // transientBulk id is of format: storeName/objectId
        final int indexOfSlash = transientBulkId.indexOf('/');
        if (indexOfSlash <= 0 || transientBulkId.length() <= indexOfSlash) {
          _log.warn("Stored transient bulk id '" + transientBulkId + "' is invalid, skipping");
        } else {
          final String storeName = transientBulkId.substring(0, indexOfSlash);
          final String objectId = transientBulkId.substring(indexOfSlash + 1);
          try {
            removeStoreObjectQuietly(storeName, objectId);
            if (_log.isDebugEnabled()) {
              _log.debug("Deleted transient bulk '" + objectId + "' from store'" + storeName + "'.");
            }
          } catch (final Exception e) {
            _log.warn("Error while deleting transient bulk object '" + objectId + "' in store '" + storeName
              + "'. It could not be deleted.", e);
          }
        }
      }
    } catch (final JobManagerException ex) {
      _log.warn("Failed to retrieve stored transient bulk Ids, obsolete data remains in stores possibly.", ex);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void checkAndHandleJobRunCompleted(final String jobName, final String jobRunId) throws JobManagerException {
    boolean hasActiveWorkflowRuns = _runStorage.checkAndCleanupActiveWorkflowRuns(jobName, jobRunId);
    if (!hasActiveWorkflowRuns && _runStorage.getJobState(jobName) == JobState.FINISHING) {
      // check if we need to create completion tasks
      final boolean ok = _runStorage.setJobState(jobName, jobRunId, JobState.FINISHING, JobState.COMPLETING);
      if (ok) {
        // only if we had a successful job run...
        if (jobRunWouldSucceed(jobName)) {
          try {
            addCompletionTaksForJobRun(jobName, jobRunId);
          } catch (final TaskGeneratorException e) {
            _log.warn("Cannot start completing workflow run due to task generator error: ", e);
            completeJobRun(jobName, jobRunId, JobState.FAILED);
            throw new JobManagerException("Could not start completing workflow run. Job run failed.", e, false);
          }
        } else {
          if (_log.isDebugEnabled()) {
            _log.debug("Cannot run completing workflow run, since job run failed.");
          }
        }
      } else {
        _log.warn("Couldn't change job state from " + JobState.FINISHING + " to " + JobState.COMPLETING
          + " for job run '" + jobRunId + "' of job '" + jobName + "'");
      }
    }
    // if we didn't add completion tasks above, we may now finish.
    // if not, this has to wait until these completion tasks are completed.
    hasActiveWorkflowRuns = _runStorage.checkAndCleanupActiveWorkflowRuns(jobName, jobRunId);
    if (!hasActiveWorkflowRuns && _runStorage.getJobState(jobName) == JobState.COMPLETING) {
      final boolean ok = _runStorage.setJobState(jobName, jobRunId, JobState.COMPLETING, JobState.CLEANINGUP);
      if (ok) {
        boolean succeeded = jobRunWouldSucceed(jobName);
        if (succeeded) {
          completeJobRun(jobName, jobRunId, JobState.SUCCEEDED);
        } else {
          completeJobRun(jobName, jobRunId, JobState.FAILED);
          removeTasksQuietly(jobName, jobRunId);
        }
      } else {
        _log.warn("Couldn't change job state from " + JobState.COMPLETING + " to " + JobState.CLEANINGUP
          + " for job run '" + jobRunId + "' of job '" + jobName + "'");
      }
    }
  }

  /**
   * Checks if there is any worker in the workflow that requests completing tasks to be generated and generate these
   * tasks.
   * 
   * @param jobName
   *          the name of the job in question
   * @param jobRunId
   *          the id of the current job run to be finished.
   * @throws JobManagerException
   *           couldn't get job definitions.
   */
  private void addCompletionTaksForJobRun(final String jobName, final String jobRunId) throws JobManagerException {

    final JobRun jobRun = ensureJobRun(jobName, jobRunId);
    final JobRunDefinitions jobDefinitions = _dataProvider.getJobRunDefinitions(jobRun.getJobName());
    final Collection<WorkflowAction> actionsRequestingCompletion =
      getActionsWithWorkersRequestingCompletion(jobDefinitions);

    // start this workflow run only if we know there may be some tasks generated.
    if (!actionsRequestingCompletion.isEmpty()) {

      // prepare run
      final String workflowRunId =
        _runStorage.startCompletionWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId());

      try {
        // get our completion tasks
        final Collection<Task> completionTasks =
          getCompletionTasksForJob(jobRun, workflowRunId, actionsRequestingCompletion);
        if (!completionTasks.isEmpty()) {
          _runStorage.startTasks(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId, completionTasks);
        }
        // if no tasks are generated we also have to finish the just started workflow run
        if (completionTasks.isEmpty()) {
          _runStorage.successfulWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId);
          _runStorage.deleteWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId);
        }
        // last step, to avoid that we have to roll it back if an error occurs in the Jobmanager before.
        try {
          _taskManager.addTasks(completionTasks);
        } catch (final TaskmanagerException e) {
          throw new JobManagerException("Could not add completion tasks for job '" + jobName + "', run id '"
            + jobRunId + "'.", e);
        }
      } catch (final JobManagerException e) {
        _runStorage.failedWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId);
        _runStorage.deleteWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId);
        throw e;
      }
    } else {
      if (_log.isDebugEnabled()) {
        _log.debug("No completing tasks requested for job '" + jobName + "'.");
      }
    }
  }

  /** checks if there are workers that need completions tasks and creates them. */
  private Collection<Task> getCompletionTasksForJob(final JobRun jobRun, final String workflowRunId,
    final Collection<WorkflowAction> actions) throws JobManagerException {
    final Collection<Task> completionTasks = new ArrayList<Task>();
    for (final WorkflowAction action : actions) {
      final WorkerDefinition workerDef = _defPersistence.getWorker(action.getWorker());
      final TaskGenerator taskGenerator = _taskGeneratorProvider.getTaskGenerator(workerDef);
      final List<Task> completionTasksForWorker =
        taskGenerator.createCompletionTasks(jobRun.getInputBucketsForAction(action),
          jobRun.getOutputBucketsForAction(action), jobRun.getParameters(action), action.getWorker());
      TaskGenerationUtil.setAdditionalTaskProperties(completionTasksForWorker, jobRun, workflowRunId, workerDef,
        action.getPosition());
      completionTasks.addAll(completionTasksForWorker);
    }
    return completionTasks;
  }

  /**
   * Get all actions from the workflow of the given job run definitions that have workers requesting some completing
   * workflow run tasks.
   */
  Collection<WorkflowAction> getActionsWithWorkersRequestingCompletion(final JobRunDefinitions jobRunDefinitions) {
    final Collection<WorkflowAction> actionsRequestingCompletionTasks = new ArrayList<WorkflowAction>();
    final Collection<WorkflowAction> workflowActions = new ArrayList<WorkflowAction>();
    workflowActions.add(jobRunDefinitions.getWorkflowDefinition().getStartAction());
    if (jobRunDefinitions.getWorkflowDefinition().getActions() != null) {
      workflowActions.addAll(jobRunDefinitions.getWorkflowDefinition().getActions());
    }
    for (final WorkflowAction action : workflowActions) {
      final WorkerDefinition workerDef = _defPersistence.getWorker(action.getWorker());
      if (workerDef.getModes().contains(WorkerDefinition.Mode.REQUESTSCOMPLETION)) {
        actionsRequestingCompletionTasks.add(action);
      }
    }
    return actionsRequestingCompletionTasks;
  }

  /** {@inheritDoc} */
  @Override
  public void deleteJobRunData(final String jobName, final String jobId) throws JobManagerException {
    final String currentJobRunId = _runStorage.getJobRunId(jobName);
    if (jobId.equals(currentJobRunId)) {
      throw new JobManagerException("Job run data of active job run cannot be deleted.");
    }
    _permStorage.deleteJobRunData(jobName, jobId);
  }

  /** {@inheritDoc} */
  @Override
  public Map<String, String> getJobRunsUsingStore(final String storeName) {
    final Map<String, String> jobRuns = new HashMap<String, String>();
    try {
      final Collection<String> currentJobs = _runStorage.getCurrentJobs();
      final Iterator<String> currentJobsIterator = currentJobs.iterator();
      while (currentJobsIterator.hasNext()) {
        final String jobName = currentJobsIterator.next();
        final String jobRunId = _runStorage.getJobRunId(jobName);
        JobRun jobRun = null;
        if (jobRunId != null) {
          try {
            jobRun = ensureJobRun(jobName, jobRunId);
          } catch (final JobManagerException e) {
            ;// do nothing, try next job name.
          }
          if (jobRun != null) {
            final Collection<Bucket> buckets = jobRun.getBuckets();
            final Iterator<Bucket> bucketIterator = buckets.iterator();
            while (bucketIterator.hasNext()) {
              final Bucket bucket = bucketIterator.next();
              if (bucket.getStoreName().equals(storeName)) {
                jobRuns.put(jobName, jobRunId);
              }
            }
          }
        }
      }
    } catch (final RunStorageException e) {
      ;// nothing to do, empty Map will be removed.
    }
    return jobRuns;
  }

  /** {@inheritDoc} */
  @Override
  public void notifiyAboutJobRunCompletion(final String jobName) {
    final JobRun jobRun = _jobRuns.get(jobName);
    if (jobRun != null) {
      final String jobRunId = jobRun.getJobRunId();
      try {
        // check if job run is completed
        if (_permStorage.containsJobRun(jobName, jobRunId)) {
          _jobRuns.remove(jobName);
        }
      } catch (final PersistenceException e) {
        // ignore errors
        _log.warn("Error while checking if job run is finished.", e);
      }
    }
  }

  @Override
  public String startWorkflowRun(final JobRun jobRun) throws JobManagerException {
    // prepare run
    final String workflowRunId = _runStorage.startWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId());
    for (final WorkflowAction barrier : jobRun.getBarrierActions()) {
      _runStorage.setupBarrier(jobRun.getJobName(), workflowRunId, barrier.getPosition());
    }
    return workflowRunId;
  }

  @Override
  public void cancelWorkflowRun(final String jobName, final String jobRunId, final String workflowRunId)
    throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("Cancel called for workflow run '" + workflowRunId + "' in job '" + jobName + "', job run '"
        + jobRunId + "'");
    }
    final JobState state = _runStorage.getJobState(jobName, jobRunId);
    checkJobRunStateForWorkflowRunCancelling(jobName, jobRunId, workflowRunId, state);
    try {
      // remove tasks in taskmanager
      final AnyMap taskFilter = DataFactory.DEFAULT.createAnyMap();
      taskFilter.put(Task.PROPERTY_JOB_NAME, jobName);
      taskFilter.put(Task.PROPERTY_JOB_RUN_ID, jobRunId);
      taskFilter.put(Task.PROPERTY_WORKFLOW_RUN_ID, workflowRunId);
      _taskManager.removeTasks(taskFilter);

      // remove transient data for canceled workflow run before canceling the workflow run in run storage,
      // otherwise transient data won't be deleted, because info about what to delete is lost.
      deleteTransientBulks(jobName, jobRunId, workflowRunId);

      // cancel workflow run and tasks in run storage
      _runStorage.cancelWorkflowRun(jobName, jobRunId, workflowRunId);

      checkAndHandleJobRunCompleted(jobName, jobRunId);

    } catch (final TaskmanagerException e) {
      throw new JobManagerException("Error while canceling workflow run " + workflowRunId + " in job run '"
        + jobRunId + "' of job '" + jobName + "': TaskManager couldn't remove canceled tasks", e);
    } catch (final Exception e) {
      throw new JobManagerException("Error while canceling workflow run " + workflowRunId + " in job run '"
        + jobRunId + "' of job '" + jobName + "'", e);
    }
  }

  /**
   * @return The readable job run id
   */
  private synchronized String createJobId() {
    final Date date = new Date();
    return String.format("%1$tY%1$tm%1$td-%1$tH%1$tM%1$tS%1$tL%2$03d", date, RANDOM.nextInt(JOB_RUN_ID_SUFFIX_MAX));
  }

  /**
   * Checks if all persistent buckets of the given job refer to an existing store. Prepares the store for all transient
   * buckets of thie given job.
   * 
   * @param jobRun
   *          the job run data
   * @throws ObjectStoreException
   *           error checking or creating store in {@link ObjectStoreService}
   * 
   */
  private void checkBuckets(final JobRun jobRun) throws ObjectStoreException {
    for (final Bucket bucket : jobRun.getBuckets()) {
      if (bucket.isTransient()) {
        ObjectStoreRetryUtil.retryEnsureStore(_objectStore, bucket.getStoreName());
      } else if (!ObjectStoreRetryUtil.retryExistsStore(_objectStore, bucket.getStoreName())) {
        throw new NoSuchStoreException("Store '" + bucket.getStoreName() + "' of persistent bucket '"
          + bucket.getBucketId() + "' does not exist.");
      }
    }
  }

  /**
   * register job to be triggered by its input buckets.
   * 
   * @param jobRun
   *          job run
   * @throws RunStorageException
   *           error
   */
  private void addJobTriggers(final JobRun jobRun) throws RunStorageException {
    final String jobName = jobRun.getJobName();
    for (final Bucket triggerBucket : jobRun.getTriggerBuckets()) {
      _runStorage.addJobTrigger(triggerBucket.getBucketId(), jobName);
    }
  }

  /**
   * prepares and automatically finishes a run once job.
   */
  private void executeRunOnceJob(final JobRun jobRun) throws JobManagerException, TaskmanagerException {
    final String workflowRunId = startWorkflowRun(jobRun);

    // in run once mode, tasks are initially read from the start action's input bucket
    final List<Task> followUpTasks = getInitialRunOnceTasks(jobRun, workflowRunId);
    if (!followUpTasks.isEmpty()) {
      _runStorage.startTasks(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId, followUpTasks);
    }
    // in run once mode, jobs are automatically finished
    finishJob(jobRun.getJobName(), jobRun.getJobRunId());
    // if no tasks are generated we also have to finish the workflow run and complete the job run
    if (followUpTasks.isEmpty()) {
      _runStorage.successfulWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId);
      _runStorage.deleteWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId);
      checkAndHandleJobRunCompleted(jobRun.getJobName(), jobRun.getJobRunId());
    }
    // last step, to avoid that we have to roll it back if an error occurs in the Jobmanager before.
    storeTasksForBarriers(jobRun, workflowRunId, followUpTasks);
    _taskManager.addTasks(followUpTasks);
  }

  private void storeTasksForBarriers(final JobRun jobRun, final String workflowRunId, final List<Task> followUpTasks)
    throws RunStorageException {
    if (jobRun.hasBarriers()) {
      final WorkflowAction action = jobRun.getStartAction();
      final Collection<WorkflowAction> barriers = jobRun.getBarriersForAction(action);
      if (barriers != null) {
        for (final WorkflowAction barrier : barriers) {
          for (final Task task : followUpTasks) {
            _runStorage.addTaskForBarrier(jobRun.getJobName(), workflowRunId, barrier.getPosition(),
              task.getTaskId());
          }
        }
      }
    }
  }

  /**
   * get initial tasks for run once job run.
   */
  private List<Task> getInitialRunOnceTasks(final JobRun jobRun, final String workflowRunId)
    throws TaskGeneratorException {
    final WorkflowAction startAction = jobRun.getStartAction();
    final WorkerDefinition workerDef = _defPersistence.getWorker(startAction.getWorker());
    final TaskGenerator taskGenerator = _taskGeneratorProvider.getTaskGenerator(workerDef);

    final AnyMap parameters = jobRun.getParameters(startAction);
    final Map<String, Bucket> inputSlotNameToBucket = jobRun.getInputBucketsForAction(startAction);
    final Map<String, Bucket> outputSlotNameToBucketMap = jobRun.getOutputBucketsForAction(startAction);

    final List<Task> tasks =
      taskGenerator.createRunOnceTasks(inputSlotNameToBucket, outputSlotNameToBucketMap, parameters,
        startAction.getWorker());
    TaskGenerationUtil.setAdditionalTaskProperties(tasks, jobRun, workflowRunId, workerDef,
      startAction.getPosition());
    return tasks;
  }

  /**
   * Clean up a failed job run. Set state to FAILED and persist the run data.
   * 
   * @param jobName
   *          job name
   * @param jobRunId
   *          job run id
   */
  private void cleanupFailedJobRun(final String jobName, final String jobRunId) {
    if (jobRunId != null) {
      try {
        if (_jobRuns.containsKey(jobName)) {
          // we have to make sure that no other job run was started in parallel
          final String currentJobRunId = _runStorage.getJobRunId(jobName);
          if (jobRunId.equals(currentJobRunId)) {
            removeJobTriggers(_jobRuns.get(jobName));
          }
        }
        completeJobRun(jobName, jobRunId, JobState.FAILED);
      } catch (final Exception ex) {
        _log.warn(
          "Error while cleaning up failed job '" + jobName + "' with run '" + jobRunId + "' after prepare.", ex);
      }
    }
  }

  /**
   * remove a job from being triggered by its input buckets.
   * 
   * @param jobRun
   *          job run
   * @throws RunStorageException
   *           error
   */
  private void removeJobTriggers(final JobRun jobRun) throws RunStorageException {
    final String jobName = jobRun.getJobName();
    for (final Bucket triggerBucket : jobRun.getTriggerBuckets()) {
      try {
        _runStorage.removeJobTrigger(triggerBucket.getBucketId(), jobName);
      } catch (final Exception ex) {
        _log.info("Could not remove job '" + jobName + "' from trigger bucket '" + triggerBucket.getBucketId()
          + "'. This is not critical.", ex);
      }
    }
  }

  /**
   * Check if given (not found) job run can be found in persistent storage to throw an appropriate exception.
   */
  private JobManagerException newExceptionNoCurrentJobRun(final String jobName, final String jobRunId)
    throws PersistenceException {
    if (!_defPersistence.hasJob(jobName)) {
      return new ConfigNotFoundException("Couldn't find job definition '" + jobName + "'.");
    }
    if (!_permStorage.containsJobRun(jobName, jobRunId)) {
      return new ConfigNotFoundException("Couldn't find job run '" + jobRunId + "' for job '" + jobName + "'.");
    }
    return new IllegalJobStateException("Requested job run '" + jobRunId + "' of job '" + jobName
      + "' was already closed, can be found in job run history.", true);
  }

  /**
   * set final state, read, persist and delete job run data.
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @param finalState
   *          final state: {@link JobState#SUCCEEDED} or {@link JobState#FAILED or {@link JobState#CANCELED}.
   * @throws JobManagerException
   *           error completing the job
   */
  private void completeJobRun(final String jobName, final String jobRunId, JobState finalState)
    throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("Cleaning up job run '" + jobRunId + "' for job '" + jobName + "' with final state " + finalState);
    }
    final AnyMap jobRunData = DataFactory.DEFAULT.createAnyMap();
    // we have to make sure that no other job run was started in parallel
    final String currentJobRunId = _runStorage.getJobRunId(jobName);
    if (jobRunId.equals(currentJobRunId)) {
      // get job run data (with details) from run storage
      jobRunData.putAll(_runStorage.getJobRunData(jobName, true));
    }
    // store job run data in persistent storage
    jobRunData.put(JobManagerConstants.DATA_JOB_STATE, finalState.name());
    jobRunData.put(JobManagerConstants.DATA_JOB_RUN_END_TIME, _runStorage.getCurrentTimestamp());
    _permStorage.storeJobRun(jobName, jobRunId, sortedJobRunData(jobRunData));

    // delete job run from run storage if no other job run was started
    if (jobRunId.equals(currentJobRunId)) {
      _runStorage.deleteJobRun(jobName, jobRunId);
      _jobRuns.remove(jobName);
    }
  }

  /** check if the job run would be successful. */
  private boolean jobRunWouldSucceed(final String jobName) throws RunStorageException {
    final AnyMap jobRunData = _runStorage.getJobRunData(jobName, false);
    // if workflow runs where started, but none was successful, the job actually failed.
    final AnyMap workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
    final int startedWorkflowRuns =
      Integer.parseInt(workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS));
    final int successfulWorkflowRuns =
      Integer.parseInt(workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
    if (startedWorkflowRuns > 0 && successfulWorkflowRuns == 0) {
      return false;
    }
    return true;
  }

  /**
   * remove object from store, do not throw exception if that fails.
   */
  private void removeStoreObjectQuietly(final String storeName, final String objectId) {
    try {
      _objectStore.removeObject(storeName, objectId);
    } catch (final Exception ex) {
      _log.warn("Error removing obsolete input object '" + objectId + "'", ex);
    }
  }

  /** re-order job run data: simple values sorted alphabetically first, complex value as ordered in input last. */
  private AnyMap sortedJobRunData(final AnyMap jobRunData) {
    final Map<String, Any> simpleValues = new TreeMap<String, Any>();
    final AnyMap complexValues = jobRunData.getFactory().createAnyMap();
    for (final Entry<String, Any> entry : jobRunData.entrySet()) {
      final String key = entry.getKey();
      final Any value = entry.getValue();
      if (value.isValue()) {
        simpleValues.put(key, value);
      } else {
        complexValues.put(key, value);
      }
    }
    final AnyMap sortedData = jobRunData.getFactory().createAnyMap();
    sortedData.putAll(simpleValues);
    sortedData.putAll(complexValues);
    return sortedData;
  }

  /**
   * check if all jobs to which dependencies exist are already running.
   * 
   * @throws JobManagerException
   *           any of the jobs, the current job is depending on is not currently running.
   */
  private void checkJobDependenciesForStarting(final JobRunDefinitions jobRunDefs) throws JobManagerException {
    final Map<String, WorkerDefinition> referredJobs = getReferredJobs(jobRunDefs);

    checkReferredJobsAreRunning(jobRunDefs, referredJobs);
  }

  /** gets the referred jobs from a {@link JobRunDefinitions} instance. */
  private Map<String, WorkerDefinition> getReferredJobs(final JobRunDefinitions jobRunDefs) {
    final Map<String, WorkerDefinition> referredJobs = new HashMap<String, WorkerDefinition>();

    final Collection<WorkflowAction> workflowActions = new ArrayList<WorkflowAction>();
    if (jobRunDefs.getWorkflowDefinition().getStartAction() != null) {
      workflowActions.add(jobRunDefs.getWorkflowDefinition().getStartAction());
    }
    if (jobRunDefs.getWorkflowDefinition().getActions() != null) {
      workflowActions.addAll(jobRunDefs.getWorkflowDefinition().getActions());
    }
    for (final WorkflowAction workflowAction : workflowActions) {
      final WorkerDefinition workerDefinition = _defPersistence.getWorker(workflowAction.getWorker());
      final Collection<ParameterDefinition> jobNameParameters =
        workerDefinition.getParametersByRange(JobManagerConstants.RANGE_JOB_NAME);
      if (!jobNameParameters.isEmpty()) {
        final AnyMap jobParameters = jobRunDefs.getJobDefinition().getParameters();
        AnyMap workflowParameters = jobRunDefs.getWorkflowDefinition().getParameters();
        final AnyMap evaluatedParameters =
          TaskParameterUtils.mergeAndEvaluateParameters(jobParameters, workflowParameters,
            workflowAction.getParameters(), workflowAction.getWorker());
        for (final ParameterDefinition jobNameParameter : jobNameParameters) {
          final String jobNameValue = evaluatedParameters.getStringValue(jobNameParameter.getName());
          if (jobNameValue != null) {
            referredJobs.put(jobNameValue, workerDefinition);
          }
        }
      }
    }
    return referredJobs;
  }

  /**
   * Checks that all jobs are running.
   * 
   * @param referredJobs
   *          the jobs that are to be checked with their referring worker.
   * @throws JobManagerException
   *           any of the jobs, the current job is depending on is not currently running. Or there had been an error
   *           checking the state of one of the jobs.
   */
  private void checkReferredJobsAreRunning(final JobRunDefinitions jobRunDefs,
    final Map<String, WorkerDefinition> referredJobs) throws JobManagerException {
    final Collection<String> referredJobsThatAreNotRunning = new ArrayList<String>();
    for (final Entry<String, WorkerDefinition> entry : referredJobs.entrySet()) {
      final String jobName = entry.getKey();
      if (_log.isDebugEnabled()) {
        _log.debug("Checking if job '" + jobName + "' is running (job '" + jobRunDefs.getJobDefinition().getName()
          + "', worker '" + entry.getValue().getName() + "'.");
      }
      try {
        final JobState state = _runStorage.getJobState(jobName);
        if (state != JobState.RUNNING && !jobName.equals(jobRunDefs.getJobDefinition().getName())) {
          referredJobsThatAreNotRunning.add(jobName);
        }
      } catch (final RunStorageException e) {
        throw new JobManagerException("Error during start of job '" + jobRunDefs.getJobDefinition().getName()
          + "'. Cannot check if referred job '" + jobName + "' is running.", e);
      }
    }
    if (referredJobsThatAreNotRunning.size() == 1) {
      throw new JobDependencyException("Error during start of job '" + jobRunDefs.getJobDefinition().getName()
        + "' because the referred job " + getJobsString(referredJobsThatAreNotRunning) + " is not running.");
    } else if (referredJobsThatAreNotRunning.size() > 1) {
      throw new JobDependencyException("Error during start of job '" + jobRunDefs.getJobDefinition().getName()
        + "' because the referred jobs " + getJobsString(referredJobsThatAreNotRunning) + " are not running.");
    }
  }

  /**
   * Check that no dependent job is still running (or more precisely is in state: PREPARING, RUNNING, FINISHING or
   * COMPLETING).
   * 
   * @param jobNameOfJobToBeFinished
   *          the name of the job that is to be finished.
   * @param jobRunIdOfJobToBeFinished
   *          the id of the job that is to be finished.
   * @throws JobManagerException
   *           either the job cannot be finished because some dependent job is still running or an exception occurred
   *           whiule checking.
   */
  private void checkNoDependentJobIsRunning(final String jobNameOfJobToBeFinished,
    final String jobRunIdOfJobToBeFinished) throws JobManagerException {
    final Collection<String> dependentJobs = new ArrayList<String>();
    for (final String jobName : _defPersistence.getJobs()) {
      final JobRunInfo jobRunInfo = _dataProvider.getJobRunInfo(jobName);
      if (jobRunInfo != null && !jobName.equals(jobNameOfJobToBeFinished)) {
        switch (jobRunInfo.getState()) {
          case PREPARING:
          case RUNNING:
          case FINISHING:
          case COMPLETING:
          case CLEANINGUP:
            if (checkIfJobIsDependent(jobName, jobNameOfJobToBeFinished)) {
              dependentJobs.add(jobName);
            }
            break;
          default:
            break;
        }
      }
    }
    if (dependentJobs.size() == 1) {
      throw new JobDependencyException("Cannot finish job '" + jobNameOfJobToBeFinished
        + "' because dependent job " + getJobsString(dependentJobs) + " is still running.");
    } else if (dependentJobs.size() > 1) {
      throw new JobDependencyException("Cannot finish job '" + jobNameOfJobToBeFinished
        + "' because dependent jobs " + getJobsString(dependentJobs) + " are still running.");
    }
  }

  /** Concatenates job names for printing/exceptions. */
  private String getJobsString(final Collection<String> dependentJobs) {
    final StringBuilder jobsStringbuilder = new StringBuilder();
    final Iterator<String> dependentJobNamesIter = dependentJobs.iterator();
    while (dependentJobNamesIter.hasNext()) {
      final String dependentJob = dependentJobNamesIter.next();
      jobsStringbuilder.append('\'').append(dependentJob).append('\'');
      if (dependentJobNamesIter.hasNext()) {
        jobsStringbuilder.append(", ");
      }
    }
    return jobsStringbuilder.toString();
  }

  /** Checks if the job with the name 'jobName' is dependent to the job with the name 'jobNameOfJobToBeFinished'. */
  private boolean checkIfJobIsDependent(final String jobName, final String jobNameOfJobToBeFinished)
    throws JobManagerException {
    final JobRunDefinitions jobRunDefs = _dataProvider.getJobRunDefinitions(jobName);
    final Map<String, WorkerDefinition> referredJobs = getReferredJobs(jobRunDefs);
    return referredJobs.containsKey(jobNameOfJobToBeFinished);
  }

  /** remove job run tasks in taskmanager. */
  private void removeTasksQuietly(final String jobName, final String jobRunId) {
    final AnyMap taskFilter = DataFactory.DEFAULT.createAnyMap();
    taskFilter.put(Task.PROPERTY_JOB_NAME, jobName);
    taskFilter.put(Task.PROPERTY_JOB_RUN_ID, jobRunId);
    try {
      _taskManager.removeTasks(taskFilter);
    } catch (TaskmanagerException e) {
      // remove quietly, so don't throw an exception here
      _log.warn("Couldn't remove taskmanager tasks for job run " + jobRunId + " of job " + jobName);
    }
  }

  /**
   * Add a RequestHandler..
   * 
   * @param listener
   *          the new RequestHandler
   */
  public void addJobListener(final JobListener listener) {
    _jobListeners.add(listener);
  }

  /**
   * Remove the given RequestHandler.
   * 
   * @param listener
   *          the RequestHandler
   */
  public void removeJobListener(final JobListener listener) {
    if (_jobListeners.contains(listener)) {
      _jobListeners.remove(listener);
    }
  }

  /** set OSGI service. */
  public void setPermanentStorage(final PermanentStorage permStorage) {
    _permStorage = permStorage;
  }

  /** unset OSGI service. */
  public void unsetPermanentStorage(final PermanentStorage permStorage) {
    if (_permStorage == permStorage) {
      _permStorage = null;
    }
  }

  /** set OSGI service. */
  public void setRunStorage(final RunStorage runStorage) {
    _runStorage = runStorage;
  }

  /**
   * @param runStorage
   *          RunStorage reference.
   */
  public void unsetRunStorage(final RunStorage runStorage) {
    if (_runStorage == runStorage) {
      _runStorage = null;
    }
  }

  /** set OSGI service. */
  public void setObjectStoreService(final ObjectStoreService objectStore) {
    _objectStore = objectStore;
  }

  /** unset OSGI service. */
  public void unsetObjectStoreService(final ObjectStoreService objectStore) {
    if (_objectStore == objectStore) {
      _objectStore = null;
    }
  }

  /** set OSGI service. */
  public void setTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    _taskGeneratorProvider = taskGeneratorProvider;
  }

  /** unset OSGI service. */
  public void unsetTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    if (_taskGeneratorProvider == taskGeneratorProvider) {
      _taskGeneratorProvider = null;
    }
  }

  /** set OSGI service. */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /** unset OSGI service. */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }

  /** set OSGI service. */
  public void setDefinitionPersistence(final DefinitionPersistence defPersistence) {
    _defPersistence = defPersistence;
  }

  /** unset OSGI service. */
  public void unsetDefinitionPersistence(final DefinitionPersistence defPersistence) {
    if (_defPersistence == defPersistence) {
      _defPersistence = null;
    }
  }

  /** set OSGI service. */
  public void setJobRunDataProvider(final JobRunDataProvider dataProvider) {
    _dataProvider = dataProvider;
  }

  /** unset OSGI service. */
  public void unsetJobRunDataProvider(final JobRunDataProvider dataProvider) {
    if (_dataProvider == dataProvider) {
      _dataProvider = null;
    }
  }

}
