/*********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.importing.crawler.file;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Set;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.importing.ContentFetcher;

/**
 * Interface for FileCrawlerService.
 * 
 * @author stuc07
 * 
 */
public interface FileCrawlerService extends ContentFetcher {

  public static final String PROPERTY_FILE_NAME = "fileName";

  public static final String PROPERTY_FILE_PATH = "filePath";

  public static final String PROPERTY_FILE_FOLDER = "fileFolder";

  public static final String PROPERTY_FILE_READ_ACL = "fileReadAccess";

  public static final String PROPERTY_FILE_WRITE_ACL = "fileWriteAccess";

  public static final String PROPERTY_FILE_SIZE = "fileSize";

  public static final String PROPERTY_FILE_LAST_MODIFIED = "fileLastModified";

  public static final String PROPERTY_FILE_EXTENSION = "fileExtension";

  public static final String ATTACHMENT_FILE_CONTENT = "fileContent";

  public static final String PRINCIPAL_OTHERS = "_OTHERS_";

  /**
   * List all files of a given directory.
   * 
   * @param directory
   * @return a Collection<File> of files
   * @throws IOException
   */
  Collection<Path> listFiles(final Path directory) throws IOException;

  /**
   * List all directories of a given directory.
   * 
   * @param directory
   * @return a Collection<File> of directories
   * @throws IOException
   */
  Collection<Path> listDirectories(final Path directory) throws IOException;

  /**
   * List all directories and files of a given directory.
   * 
   * @param directory
   * @return a Collection<File> of directories and files
   * @throws IOException
   */
  Collection<Path> list(final Path directory) throws IOException;

  /**
   * Create a Record for the given file and dataSource and fill it with all file properties. Optionally also fill with
   * content attachment.
   * 
   * @param file
   * @param dataSource
   * @param withContent
   *          if true fill also content attachment, otherwise not
   * @return a Record
   * @throws IOException
   */
  Record fileToRecord(final Path file, final String dataSource, final boolean withContent) throws IOException;

  /**
   * Create a Record for the given directory and dataSource and fill it with ImportingConstants.ROOT_FOLDER_ATTRIBUTE.
   * 
   * @param directory
   * @param dataSource
   * @return a record
   * @throws IOException
   */
  Record dirToRecord(final Path directory, final String dataSource) throws IOException;

  /**
   * Add the content attachment to the given record.
   * 
   * @param record
   * @param pathAttribute
   *          name of the attribute containing the path to the file
   * @param attachmentName
   *          name with which the attachment is stored in the recor
   * @throws IOException
   */
  void addAttachment(final Record record, final String pathAttribute, final String attachmentName)
    throws IOException;

  /**
   * Returns a Set containing all file property and attachment names.
   * 
   * @return a Set of String
   */
  Set<String> getFilePropertyNames();
}
