/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.definitions;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.smila.common.definitions.AccessAny;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Value;

/**
 * An action definition within a workflow.
 */
public class WorkflowAction {
  /** name of JSON property for worker. */
  public static final String KEY_WORKER = "worker";

  /** name of JSON property for parameters. */
  public static final String KEY_PARAMETERS = "parameters";

  /** name of JSON property for input bucket mapping. */
  public static final String KEY_INPUT = "input";

  /** name of JSON property for output bucket mapping. */
  public static final String KEY_OUTPUT = "output";

  /** position in workflow definition, 0 = start action. */
  private final int _position;

  /** worker is mandatory. */
  private final String _worker;

  /** parameters are optional. */
  private final AnyMap _parameters;

  /** input is optional. */
  private final Map<String, String> _input;

  /** output is optional. */
  private final Map<String, String> _output;

  /**
   * parse workflow action definition from Any object.
   * 
   * @param workflowAction
   *          see class comment for format of Any object
   * @throws Exception
   *           An exception that occurred while converting the object.
   */
  public WorkflowAction(final AnyMap workflowAction, final int position) throws Exception {
    _position = position;
    _worker = AccessAny.getStringRequired(workflowAction, KEY_WORKER);
    _parameters = (AnyMap) AccessAny.get(workflowAction, KEY_PARAMETERS, ValueType.MAP);
    final AnyMap inputAny = (AnyMap) AccessAny.get(workflowAction, KEY_INPUT, ValueType.MAP);
    if (inputAny != null) {
      _input = new LinkedHashMap<String, String>();
      for (final Entry<String, Any> entry : inputAny.entrySet()) {
        _input.put(entry.getKey(), ((Value) entry.getValue()).asString());
      }
    } else {
      _input = null;
    }
    final AnyMap outputAny = (AnyMap) AccessAny.get(workflowAction, KEY_OUTPUT, ValueType.MAP);
    if (outputAny != null) {
      _output = new LinkedHashMap<String, String>();
      for (final Entry<String, Any> entry : outputAny.entrySet()) {
        _output.put(entry.getKey(), ((Value) entry.getValue()).asString());
      }
    } else {
      _output = null;
    }
  }

  /**
   * Returns the object as an Any representation.
   * 
   * @return Any object describing this workflow action definition.
   * @throws Exception
   *           An exception that occurred while converting the object.
   */
  public Any toAny() throws Exception {
    final AnyMap workflowActionAny = AccessAny.FACTORY.createAnyMap();
    workflowActionAny.put(KEY_WORKER, _worker);
    if (_parameters != null) {
      workflowActionAny.put(KEY_PARAMETERS, _parameters);
    }
    if (_input != null) {
      final AnyMap inputAny = AccessAny.FACTORY.createAnyMap();
      for (final Entry<String, String> entry : _input.entrySet()) {
        inputAny.put(entry.getKey(), entry.getValue());
      }
      workflowActionAny.put(KEY_INPUT, inputAny);
    }
    if (_output != null) {
      final AnyMap outputAny = AccessAny.FACTORY.createAnyMap();
      for (final Entry<String, String> entry : _output.entrySet()) {
        outputAny.put(entry.getKey(), entry.getValue());
      }
      workflowActionAny.put(KEY_OUTPUT, outputAny);
    }

    return workflowActionAny;
  }

  /** @return position in workflow. */
  public int getPosition() {
    return _position;
  }

  /** @return the worker */
  public String getWorker() {
    return _worker;
  }

  /** @return the parameters */
  public AnyMap getParameters() {
    return AccessAny.cloneAnyMap(_parameters);
  }

  /** @return the input */
  public Map<String, String> getInput() {
    return AccessAny.copyMap(_input);
  }

  /** @return the output */
  public Map<String, String> getOutput() {
    return AccessAny.copyMap(_output);
  }

  /** {@inheritDoc} */
  @Override
  public String toString() {
    try {
      return this.toAny().toString();
    } catch (final Exception ex) {
      return String.format("%s %s%n\t%s %s%n\t%s %s%n\t%s %s", KEY_WORKER, _worker, KEY_PARAMETERS, _parameters,
        KEY_INPUT, _input, KEY_OUTPUT, _output);
    }
  }

  /** {@inheritDoc} */
  @Override
  public int hashCode() {
    int hash = getWorker().hashCode();
    if (getInput() != null) {
      hash += getInput().hashCode();
    }
    if (getOutput() != null) {
      hash += getOutput().hashCode();
    }
    if (getParameters() != null) {
      hash += getParameters().hashCode();
    }
    return hash;
  }

  /** {@inheritDoc} */
  @Override
  public boolean equals(final Object o) {
    if (o == this) {
      return true;
    }
    if (!(o instanceof WorkflowAction)) {
      return false;
    }
    final WorkflowAction wa = (WorkflowAction) o;

    if (!wa.getWorker().equals(this.getWorker())) {
      return false;
    }

    boolean inputEqual = false;
    if (wa.getInput() != null) {
      inputEqual = wa.getInput().equals(this.getInput());
    } else {
      inputEqual = wa.getInput() == null;
    }
    boolean outputEqual = false;
    if (wa.getOutput() != null) {
      outputEqual = wa.getOutput().equals(this.getOutput());
    } else {
      outputEqual = wa.getOutput() == null;
    }
    boolean paramEqual = false;
    if (wa.getParameters() != null) {
      paramEqual = wa.getParameters().equals(this.getParameters());
    } else {
      paramEqual = wa.getParameters() == null;
    }

    return inputEqual && outputEqual && paramEqual;
  }
}
