/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.common.definitions.DefinitionBase;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpExchange;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.definitions.BucketDefinition;
import org.eclipse.smila.jobmanager.definitions.DefinitionPersistence;
import org.eclipse.smila.jobmanager.definitions.JobManagerConstants;

/**
 * Implements the handling of HTTP requests to request the buckets list or to define new buckets. <br>
 * 
 * URL pattern: <code>smila/jobmanager/buckets/</code> <br>
 * 
 * Methods allowed: <code>GET, POST</code>
 */
public class BucketsHandler extends AJobManagerHandler {

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord,
    final HttpExchange exchange) throws Exception {
    final DefinitionPersistence defPersistence = getDefinitionPersistence();
    if (isGetRequest(method)) {
      final Map<String, Object> resultMap = new HashMap<String, Object>();
      final List<Map<String, Object>> bucketList = new ArrayList<Map<String, Object>>();
      for (final String bucketName : defPersistence.getBuckets()) {
        final LinkedHashMap<String, Object> map = new LinkedHashMap<String, Object>();
        map.put("name", bucketName);
        map.put("url", URLCreator.create(getRequestHost(exchange), requestUri, bucketName));
        bucketList.add(map);
      }
      resultMap.put(BucketDefinition.KEY_BUCKETS, bucketList);
      return resultMap;
    }
    if (isPostRequest(method)) {
      if (inputRecord == null) {
        throw new IllegalArgumentException("Missing input, cannot create buckets from nothing.");
      }
      final AnyMap definition = inputRecord.getMetadata();
      // create timestamp for this workflow and store it with the data
      final Any timestamp = inputRecord.getFactory().createDateTimeValue(new Date());
      definition.put(DefinitionBase.KEY_TIMESTAMP, timestamp);
      // remove readOnly flag if present.
      definition.remove(DefinitionBase.KEY_READ_ONLY);
      final BucketDefinition createdBucket = BucketDefinition.parseBucket(definition);
      defPersistence.addBucket(createdBucket);
      final AnyMap result = DataFactory.DEFAULT.createAnyMap();
      result.put("name", createdBucket.getName());
      result.put(JobManagerConstants.TIMESTAMP, timestamp);
      result.put("url", URLCreator.create(getRequestHost(exchange), requestUri, createdBucket.getName())
        .toExternalForm());
      return result;
    }
    return null;
  }

  /**
   * result status for POST requests should be CREATED.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    if ("POST".equals(method)) {
      return HttpStatus.CREATED;
    }
    return super.getSuccessStatus(method, requestUri);
  }

  /**
   * {@inheritDoc}
   * 
   * GET and POST are currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return isGetOrPostRequest(method);
  }
}
