/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.taskgenerator;

import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.definitions.Bucket;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;

/**
 * Interface for TaskGenerator implementations.
 * 
 * TaskGenerators are used to create (follow-up) tasks for a specifc workflow action/worker.
 */
public interface TaskGenerator {

  /**
   * @return name of generator, it is used in Worker descriptions to select the generator.
   */
  String getName();

  /**
   * Create tasks based on additional job run information.
   * 
   * @param changedInput
   *          key: slot containing added bulks, value: added bulks
   * @param inputBuckets
   *          key: input slot name, value: bucket for this slot
   * @param outputBuckets
   *          key: output slot name, value: bucket for this slot
   * @param parameters
   *          key: param name, value: (evaluated) param value
   * @param workerName
   *          (action) worker for which to create new tasks
   * @param jobName
   *          name of current job
   * @param jobRunId
   *          id of current job run
   * @param workflowRunId
   *          id of current workflow run
   * 
   * @return a list of new generated tasks
   * @throws TaskGeneratorException
   *           error while generating tasks
   */
  List<Task> createTasks(final Map<String, List<BulkInfo>> changedInput, final Map<String, Bucket> inputBuckets,
    final Map<String, Bucket> outputBuckets, final AnyMap parameters, final String workerName,
    final String jobName, String jobRunId, final String workflowRunId) throws TaskGeneratorException;

  /**
   * @param changedInput
   *          key: slot containing added bulks, value: added bulks
   * @param inputBuckets
   *          key: input slot name, value: bucket for this slot
   * @param outputBuckets
   *          key: output slot name, value: bucket for this slot
   * @param parameters
   *          key: param name, value: (evaluated) param value
   * @param workerName
   *          (action) worker for which to create new tasks
   * @return a list of new generated tasks
   * @throws TaskGeneratorException
   *           error while generating tasks
   */
  List<Task> createTasks(final Map<String, List<BulkInfo>> changedInput, final Map<String, Bucket> inputBuckets,
    final Map<String, Bucket> outputBuckets, final AnyMap parameters, final String workerName)
    throws TaskGeneratorException;

  /**
   * @param task
   *          The task to be cleaned up
   * @param taskState
   *          TODO
   * 
   * @throws TaskGeneratorException
   *           error while cleaning up for finished task
   */
  void finishTask(final Task task, TaskCompletionStatus taskState) throws TaskGeneratorException;

  /**
   * Creates tasks for the RunOnce mode, i.e. for all objects in the input bucket of the start action.
   * 
   * @param inputBuckets
   *          key: input slot name, value: bucket for this slot
   * @param outputBuckets
   *          key: output slot name, value: bucket for this slot
   * @param parameters
   *          key: param name, value: (evaluated) param value
   * @param workerName
   *          (action) worker for which to create new tasks
   * @return a list of new generated tasks
   * @throws TaskGeneratorException
   *           error while generating tasks
   */
  List<Task> createRunOnceTasks(final Map<String, Bucket> inputBuckets, final Map<String, Bucket> outputBuckets,
    final AnyMap parameters, final String workerName) throws TaskGeneratorException;

  /**
   * Creates tasks for the completion workflow run, i.e. the run just before the job run is completed to to some
   * cleanup/completion work.
   * 
   * @param inputBuckets
   *          key: input slot name, value: bucket for this slot
   * @param outputBuckets
   *          key: output slot name, value: bucket for this slot
   * @param parameters
   *          key: param name, value: (evaluated) param value
   * @param workerName
   *          (action) worker for which to create new tasks
   * @return a list of new generated tasks
   * @throws TaskGeneratorException
   *           error while generating tasks
   */
  List<Task> createCompletionTasks(final Map<String, Bucket> inputBuckets, final Map<String, Bucket> outputBuckets,
    final AnyMap parameters, final String workerName) throws TaskGeneratorException;

  /**
   * Check complex parameters in some task generators if desired. Do nothing if no check desired.
   * 
   * @param parameters
   *          The map with the parameters
   * @throws TaskGeneratorException
   *           an exception if something is wrong with the parameters
   */
  void validateParameters(final AnyMap parameters) throws TaskGeneratorException;
}
