/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.datamodel.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Attachment;
import org.eclipse.smila.datamodel.Record;

/** Helper class for merging records. */
public final class RecordMerger {

  /**
   * merge metadata and attachments of record 'from' into record 'to'.
   * 
   * @param mergeExistingAttributes
   *          if 'true', the values of equal metadata attributes in record 'to' and 'from' will be merged (create AnySeq
   *          of values resp. merge existing AnySeq's). If 'false', equal metadata attributes already exist in record
   *          'to' will be overwritten, only attribute recordid will be kept if it was set before.
   * @param mergeExistingAttachments
   *          if 'true', equal attachments in record 'to' and 'from' will be merged (binary data is concatenated). If
   *          'false', equal attachments already exist in record 'to' will be overwritten.
   */
  public static void mergeRecords(final Record to, final Record from, final boolean mergeExistingAttributes,
    final boolean mergeExistingAttachments) {
    final String toId = to.getId();
    if (mergeExistingAttributes) {
      mergeMetadata(to.getMetadata(), from.getMetadata());
    } else {
      to.getMetadata().putAll(from.getMetadata());
    }
    if (toId != null) {
      to.setId(toId); // if record id was set, it must be kept
    }

    final Iterator<String> attachmentNames = from.getAttachmentNames();
    while (attachmentNames.hasNext()) {
      final String attachName = attachmentNames.next();
      if (mergeExistingAttachments) {
        to.setAttachment(attachName, mergeAttachments(attachName, from, to));
      } else {
        to.setAttachment(from.getAttachment(attachName));
      }
    }
  }

  /** Helper function to concatenate attachments in the order "to, from" */
  private static byte[] mergeAttachments(String attachName, Record from, Record to) {
    if (to.hasAttachment(attachName)) {
      final Attachment toAttachment = to.getAttachment(attachName);
      final Attachment fromAttachment = from.getAttachment(attachName);
      try {
        final ByteArrayOutputStream concatAttachment = new ByteArrayOutputStream();
        concatAttachment.write(toAttachment.getAsBytes());
        concatAttachment.write(fromAttachment.getAsBytes());
        return concatAttachment.toByteArray();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    // if attachment was not contained in "to" yet or an error happens default to use the from attachment 
    return from.getAttachmentAsBytes(attachName);
  }

  /** helper method to merge metadata attributes. TODO do this recursive, not only on top level */
  private static void mergeMetadata(final AnyMap to, final AnyMap from) {
    for (final Map.Entry<String, Any> entry : from.entrySet()) {
      final String fromKey = entry.getKey();
      if (to.containsKey(fromKey)) {
        final Any resultValue = to.get(fromKey);
        final Any valuesToMerge = entry.getValue();
        if (valuesToMerge.isValue() || valuesToMerge.isSeq()) {
          if (resultValue.isSeq()) {
            resultValue.asSeq().addAll(valuesToMerge.asSeq());
          } else if (resultValue.isValue()) {
            final AnySeq newSeq = resultValue.asSeq();
            newSeq.addAll(valuesToMerge.asSeq());
            to.put(fromKey, newSeq);
          }
        } else if (resultValue.isMap() && valuesToMerge.isMap()) {
          mergeMaps(resultValue.asMap(), valuesToMerge.asMap());
        }
      } else {
        // merge values, that are not yet contained in the target map 'to'
        to.put(fromKey, from.get(fromKey));
      }
    }
  }

  /** helper method to merge metadata attributes. */
  private static void mergeMaps(final AnyMap to, final AnyMap from) {
    // TODO also merge map values?
    to.putAll(from);
  }
}
