/*******************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.http.client.impl;

import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.HeaderElement;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.util.EntityUtils;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.ipc.IpcAnyReader;
import org.eclipse.smila.http.client.RestException;
import org.eclipse.smila.http.client.impl.base.BulkResponseImpl;
import org.eclipse.smila.http.client.impl.base.HttpRequestFactory;
import org.eclipse.smila.http.client.impl.base.HttpResultHandler;

/** Helper class for {@link DefaultRestClient} to handle results of HTTP requests. */
public class DefaultHttpResultHandler implements HttpResultHandler {
  /** Converter for JSON to Anys. */
  private final IpcAnyReader _jsonReader = new IpcAnyReader();

  /** The logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.eclipse.smila.http.client.impl.HttpResultHandler#processHttpResponse(org.apache.http.client.methods.HttpUriRequest
   * , org.apache.http.HttpResponse)
   */
  @Override
  public InputStream processHttpResponse(final HttpUriRequest request, final HttpResponse response)
    throws IOException, RestException {
    InputStream content = null;
    boolean requestCompleted = false;
    if (response != null) {
      try {
        final int httpStatus = response.getStatusLine().getStatusCode();
        if (httpStatus == HttpStatus.SC_NO_CONTENT) {
          _log.info("Request succeeded with no-content response.");
          return null;
        }
        final String responseMimeType = getResponseMimeType(response);
        if (HttpRequestFactory.MIMETYPE_JSON.equalsIgnoreCase(responseMimeType)) {
          if (response.getEntity() != null) {
            content = response.getEntity().getContent();
          }
          if (httpStatus >= HttpStatus.SC_BAD_REQUEST) {
            throw newRestException(httpStatus, response, content);
          }
        } else {
          processNonJsonResponse(response, httpStatus);
        }
        requestCompleted = true;
      } finally {
        if (!requestCompleted) {
          request.abort();
        }
      }
    }
    return content;
  }

  /** Write non-JSON response to logger, throw exception if it was an error response. */
  private void processNonJsonResponse(final HttpResponse response, final int httpStatus) throws IOException,
    RestException {
    final HttpEntity entity = response.getEntity();
    final String responseBody = getResponseBodyAsString(entity);
    if (httpStatus >= HttpStatus.SC_BAD_REQUEST) {
      if (_log.isErrorEnabled()) {
        if (responseBody != null) {
          _log.info("Request failed with non-JSON response: " + responseBody);
        } else {
          _log.info("Request failed with empty non-JSON response");
        }
      }
      throw newRestException(httpStatus, response, null);
    } else {
      if (_log.isInfoEnabled()) {
        if (responseBody != null) {
          _log.info("Request succeeded with non-JSON response: " + responseBody);
        } else {
          _log.info("Request succeeded with empty non-JSON response");
        }
      }
    }
  }

  private String getResponseBodyAsString(final HttpEntity entity) throws IOException {
    if (entity == null) {
      return null;
    }
    final String responseBody = EntityUtils.toString(entity); // in any case consume entity.
    if (entity.getContentLength() == 0) {
      return null;
    }
    return responseBody;
  }

  /** Get mime type specification from response header. */
  private String getResponseMimeType(final HttpResponse response) {
    String responseContentType = null;
    final Header contentTypeHeader = response.getFirstHeader("Content-Type");
    if (contentTypeHeader != null) {
      final HeaderElement[] elements = contentTypeHeader.getElements();
      if (elements != null && elements.length > 0) {
        responseContentType = elements[0].getName();
      }
    }
    return responseContentType;
  }

  /** Create a {@link RestException} from the HTTP response. */
  private RestException newRestException(final int statusCode, final HttpResponse response,
    final InputStream content) throws IOException {
    if (content != null) {
      Any errorObject = null;
      try {
        errorObject = handleJsonResult(content);
      } catch (final Exception ex) {
        _log.error("Failed to parse an error result", ex);
      }
      if (errorObject != null && errorObject.isMap()) {
        return new RestException(statusCode, errorObject.asMap());
      }
    }
    return new RestException(statusCode, "Error Code: " + statusCode + " "
      + response.getStatusLine().getReasonPhrase() + ", See log for details.");
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.smila.http.client.impl.HttpResultHandler#handleJsonResult(java.io.InputStream)
   */
  @Override
  public Any handleJsonResult(final InputStream content) throws IOException {
    if (content != null) {
      try {
        return _jsonReader.readJsonStream(content);
      } catch (final EOFException ex) {
        ; // caused by empty JSON result. Ignore.
      } finally {
        IOUtils.closeQuietly(content);
      }
    }
    return null;
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.smila.http.client.impl.HttpResultHandler#handleJsonBulkResult(java.io.InputStream)
   */
  @Override
  public BulkResponseImpl handleJsonBulkResult(InputStream content) throws IOException {
    boolean isContentConsumed = false;
    if (content == null) {
      content = new ByteArrayInputStream(new byte[0]);
    }
    try {
      final BulkResponseImpl bulk = new BulkResponseImpl(content);
      isContentConsumed = true;
      return bulk;
    } finally {
      if (!isContentConsumed) {
        IOUtils.closeQuietly(content);
      }
    }
  }

}
