/**********************************************************************************************************************
 * Copyright (c) 2008, 2014 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: Juergen Schumacher (Empolis Information Management GmbH) - initial implementation
 **********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.scripting.ScriptExecutor;
import org.eclipse.smila.scripting.ScriptNotFoundException;
import org.eclipse.smila.scripting.ScriptingEngineException;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Function;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.Undefined;
import org.mozilla.javascript.commonjs.module.ModuleScript;
import org.mozilla.javascript.commonjs.module.ModuleScriptProvider;
import org.mozilla.javascript.commonjs.module.Require;
import org.mozilla.javascript.commonjs.module.RequireBuilder;

/**
 * Loads a Javascript script and executes functions.
 * 
 * Note: MUST be used by a single thread only, MUST be closed after use.
 */
public class JavascriptExecutor implements AutoCloseable, ScriptExecutor {

  private final ModuleScriptProvider _scriptProvider;

  private final Scriptable _baseScope;

  private Context _context;

  private Scriptable _scriptScope;

  private String _scriptFile;

  private final Log _log = LogFactory.getLog(getClass());

  /** create instance. */
  public JavascriptExecutor(final ModuleScriptProvider scriptProvider, final Scriptable baseScope) {
    _scriptProvider = scriptProvider;
    _baseScope = baseScope;

    initScriptScope();
  }

  private void initScriptScope() {
    _context = Context.enter();
    _scriptScope = _context.newObject(_baseScope);
    _scriptScope.setPrototype(_baseScope);
    _scriptScope.setParentScope(null);

    initRequire();

    // allow that called script can contain "exports" statement. In required scripts the "exports" object
    // is made available by require(). Here we just define it, although it will not be used/needed for our call.
    _scriptScope.put("exports", _scriptScope, _context.newObject(_scriptScope));
  }

  private void initRequire() {
    final RequireBuilder requireBuilder = new RequireBuilder();
    requireBuilder.setModuleScriptProvider(_scriptProvider);
    final Require require = requireBuilder.createRequire(_context, _scriptScope);
    require.install(_scriptScope);
  }

  @Override
  public Object loadScript(final String scriptFile) throws ScriptingEngineException {
    long compileEndTime = 0;
    final long startTime = System.nanoTime();
    _scriptFile = scriptFile;
    try {
      final ModuleScript script = _scriptProvider.getModuleScript(_context, scriptFile, null, null, null);
      compileEndTime = System.nanoTime();
      if (script == null) {
        throw new ScriptNotFoundException("JavaScript script '" + _scriptFile + "' cannot be loaded.");
      }
      return script.getScript().exec(_context, _scriptScope);
    } catch (final ScriptingEngineException ex) {
      throw ex;
    } catch (final Exception ex) {
      throw new ScriptingEngineException("Error loading " + _scriptFile, ex);
    } finally {
      if (_log.isDebugEnabled()) {
        final double compileTime = (compileEndTime - startTime) / 1e6;
        final double execTime = (System.nanoTime() - compileEndTime) / 1e6;
        _log.debug(String.format(Locale.ENGLISH, "loadScript times (ms):  Compile %.3f, Exec %.3f", compileTime,
          execTime));
      }
    }
  }

  @Override
  public Record call(final String scriptFunction, final Record record) throws ScriptingEngineException {
    final Function function = getFunction(scriptFunction);
    try {
      final RecordWrapper wrapper = new RecordWrapper(record, _baseScope);
      final Object result = function.call(_context, _scriptScope, _scriptScope, new Object[] { wrapper });
      return getResultRecord(result);
    } catch (final Exception ex) {
      throw new ScriptingEngineException("Error executing function " + scriptFunction + " in " + _scriptFile, ex);
    }
  }

  @Override
  public AnyMap call(final String scriptFunction, final AnyMap arguments) throws ScriptingEngineException {
    final Function function = getFunction(scriptFunction);
    try {
      final AnyMapWrapper wrapper = new AnyMapWrapper(arguments, _baseScope);
      final Object result = function.call(_context, _scriptScope, _scriptScope, new Object[] { wrapper });
      if (result != null) {
        final Record resultAsRecord = getResultRecord(result);
        if (resultAsRecord != null) {
          return resultAsRecord.getMetadata();
        }
      }
      return null;
    } catch (final Exception ex) {
      throw new ScriptingEngineException("Error executing function " + scriptFunction + " in " + _scriptFile, ex);
    }
  }

  @Override
  public void install(final Installable plugin) throws ScriptingEngineException {
    plugin.install(_scriptScope);
  }

  private Function getFunction(final String scriptFunction) throws ScriptingEngineException {
    final Object function = _scriptScope.get(scriptFunction, _scriptScope);
    if (function == null || !(function instanceof Function)) {
      throw new ScriptNotFoundException("Script " + _scriptFile + " does not define function '" + scriptFunction
        + "'");
    }
    return (Function) function;
  }

  private Record getResultRecord(final Object result) {
    if (result == null || result instanceof Undefined) {
      return null;
    }
    final Record[] resultRecords = RecordWrapper.asRecordArray(result);
    switch (resultRecords.length) {
      case 0:
        return null;
      case 1:
        return resultRecords[0];
      default:
        final Record resultRecord = DataFactory.DEFAULT.createRecord();
        final AnySeq recordSequence = resultRecord.getMetadata().getSeq("result", true);
        for (final Record resultRecord2 : resultRecords) {
          recordSequence.add(resultRecord2.getMetadata());
        }
        return resultRecord;
    }
  }

  @Override
  public void close() {
    try {
      Context.exit();
    } catch (final Exception ex) {
      _log.info("Exception while closing executor for script " + _scriptFile, ex);
    }
  }

}
