/*********************************************************************************************************************
 * Copyright (c) 2008, 2015 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved.
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
**********************************************************************************************************************/
package org.eclipse.smila.scripting.internal;

import java.net.URI;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.processing.PipeletTracker;
import org.eclipse.wst.jsdt.debug.rhino.debugger.RhinoDebugger;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.ContextFactory;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;
import org.mozilla.javascript.commonjs.module.ModuleScriptProvider;
import org.mozilla.javascript.commonjs.module.provider.ModuleSourceProvider;
import org.mozilla.javascript.commonjs.module.provider.StrongCachingModuleScriptProvider;
import org.mozilla.javascript.commonjs.module.provider.UrlModuleSourceProvider;
import org.mozilla.javascript.tools.shell.Global;
import org.osgi.framework.BundleContext;

/**
 * Some helper functions around Rhino classes to hide details.
 */
public final class RhinoUtils {

  private static final String SCRIPT_LOG_CATEGORY = JavascriptEngine.BUNDLE_NAME + ".js";

  private RhinoUtils() {
    throw new UnsupportedOperationException("RhinoUtils must not be instantiated.");
  }

  /**
   * Print some information about the Rhino engine to the given log.
   * 
   * @param log
   *          logger to use.
   */
  public static void printEngineInfo(final Log log) {
    final Context cx = Context.enter();
    try {
      log.info("Javascript engine is " + cx.getImplementationVersion());
      log.info("Language version " + cx.getLanguageVersion());
      log.info("Optimization level " + cx.getOptimizationLevel());
      log.info("Interpreter locale " + cx.getLocale());
    } finally {
      Context.exit();
    }
  }

  /**
   * Creates and start the Eclipse JSDT debugger interface for Rhino.
   * 
   * @param connectionString
   *          see https://wiki.eclipse.org/JSDT/Debug/Rhino/Embedding_Rhino_Debugger#The_Connection_String
   * @throws Exception
   *           error starting the debugger.
   */
  public static void startDebugger(final String connectionString) throws Exception {
    final RhinoDebugger debugger = new RhinoDebugger(connectionString);
    debugger.start();
    ContextFactory.getGlobal().addListener(debugger);
  }

  /**
   * Creates a ModuleScriptProvider that loads scripts from the given directory only.
   * 
   * @param scriptDirectory
   *          base script directory.
   * @return script provider.
   */
  public static ModuleScriptProvider createScriptProvider(final Path scriptDirectory) {
    final List<URI> scriptDirsAsURIs = Arrays.asList(scriptDirectory.toUri());
    final ModuleSourceProvider sourceProvider = new UrlModuleSourceProvider(scriptDirsAsURIs, null, null, null);
    return new StrongCachingModuleScriptProvider(sourceProvider);
  }

  /**
   * Creates a Rhino base scope for SMILA script processing. The scope will already contain:
   * <ul>
   * <li>the Rhino Shell Globals as provided by {@link Global}.</li>
   * <li>a {@link PipeletBuilder} as "pipelets".</li>
   * <li>a {@link ServiceAccessor} as "services".</li>
   * <li>a default logger as "log"</li>
   * <li>a LogFactory as "LogFactory".</li>
   * 
   * The scope will be sealed so that it cannot be modified by scripts.
   * 
   * @param bundleContext
   *          bundle context for lookup of service references.
   * @param pipeletTracker
   *          the pipelet tracker service.
   * @param blackboardFactory
   *          the blackboard factory service, needed to support pipelet invocations.
   * @return
   */
  public static Scriptable createBaseScope(final BundleContext bundleContext, final PipeletTracker pipeletTracker,
    final BlackboardFactory blackboardFactory) {

    final Context context = Context.enter();
    try {
      final ScriptableObject baseScope = context.initStandardObjects(null, true);
      final Global global = new Global(context);
      baseScope.setPrototype(global);

      final PipeletBuilder pipeletBuilder = new PipeletBuilder(pipeletTracker, blackboardFactory, baseScope);
      final ServiceAccessor serviceAccessor = new ServiceAccessor(bundleContext);
      final Log scriptLog = LogFactory.getLog(SCRIPT_LOG_CATEGORY);
      final LogFactory logFactory = LogFactory.getFactory();

      ScriptableObject.putProperty(baseScope, "pipelets", Context.javaToJS(pipeletBuilder, baseScope));
      ScriptableObject.putProperty(baseScope, "services", Context.javaToJS(serviceAccessor, baseScope));
      ScriptableObject.putProperty(baseScope, "log", Context.javaToJS(scriptLog, baseScope));
      ScriptableObject.putProperty(baseScope, "LogFactory", Context.javaToJS(logFactory, baseScope));

      baseScope.sealObject();
      return baseScope;
    } finally {
      Context.exit();
    }
  }
}
