/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.search.servlet;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.fileupload.FileItem;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.search.api.helper.QueryBuilder;
import org.eclipse.smila.utils.collections.MultiValueMap;

/**
 * request parser that can parse multi part forms containing attachments.
 *
 * @author jschumacher
 *
 */
public class MultiPartRequestParser extends ARequestParser {

  private final MultiValueMap<String, String> parameters = new MultiValueMap<String, String>();

  private final Map<String, byte[]> attachments = new HashMap<String, byte[]>();

  private final AnyMap attachmentFileNames = DataFactory.DEFAULT.createAnyMap();

  /**
   * create new instance with default pipeline.
   *
   * @param defaultPipeline
   *          default pipeline name to use, if request does not contain a pipeline parameter.
   */
  public MultiPartRequestParser() {

  }

  /**
   * delivers first value of given param name. If any. In all other causes, returns null.
   *
   * @param paramName
   *          parameter name.
   */
  public String getSingleParameterValue(final String paramName) {
    String result = null;
    if (parameters.containsKey(paramName)) {
      result = parameters.get(paramName).get(0);
    }
    return result;
  }

  /**
   * initial parse of the content go get all fields seperated by attachment.
   *
   * @param items
   *          multi part from items
   * @return new query builder instance.
   */
  public void initialParse(final List<FileItem> items) {

    for (int i = 0; i < items.size(); i++) {
      final FileItem item = items.get(i);
      final String fieldName = item.getFieldName();
      if (item.isFormField()) {
        parameters.add(fieldName, item.getString());
      } else {
        if (item.getSize() > 0) {
          attachments.put(fieldName, item.get());
          attachmentFileNames.put(fieldName, DataFactory.DEFAULT.createStringValue(item.getName()));
        }
      }
    }
  }

  /**
   * create QueryBuilder from http request parameters.
   *
   * @return new query builder instance.
   */
  public QueryBuilder parse() {

    final QueryBuilder builder = new QueryBuilder();

    for (final String paramName : parameters.keySet()) {
      final List<String> values = parameters.get(paramName);
      if (values != null && values.size() > 0) {
        processParameter(builder, paramName, values.toArray(new String[values.size()]));
      }
    }

    for (final String attachmentName : attachments.keySet()) {
      final byte[] content = attachments.get(attachmentName);
      if (content != null && content.length > 0) {
        builder.setAttachment(attachmentName, content);
      }
    }

    if (!attachmentFileNames.isEmpty()) {
      builder.getQuery().getMetadata().put("attachmentFileNames", attachmentFileNames);
    }
    setupQuery(builder);
    return builder;
  }
}
