/***********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * <ul>
 * <li>Andreas Weber (Empolis Information Management GmbH) - initial API and implementation
 * <li>Peter Palmar (Empolis Information Management GmbH) - Add pushing to remote SMILA REST API.
 * </ul>
 **********************************************************************************************************************/
package org.eclipse.smila.importing.worker;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.validation.InvalidRecordException;
import org.eclipse.smila.http.client.RestClient;
import org.eclipse.smila.http.client.RestException;
import org.eclipse.smila.http.client.impl.failover.FailoverRestClient;
import org.eclipse.smila.importing.DeltaException;
import org.eclipse.smila.importing.DeltaImportStrategy;
import org.eclipse.smila.importing.DeltaService;
import org.eclipse.smila.importing.ImportingConstants;
import org.eclipse.smila.importing.ImportingException;
import org.eclipse.smila.importing.State;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.input.Inputs;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.Outputs;
import org.eclipse.smila.taskworker.output.RecordOutput;

/**
 * Worker that pushes input records to the bulkbuilder and marks the records as updated in delta indexing.
 */
public class UpdatePusherWorker extends WorkerUsingDeltaService {

  /** the {@link DeltaService} shard prefix to get records to delete from. */
  public static final String DELETE_SHARD_PARAM = "deleteRecordsShard";

  /** worker's name. */
  private static final String WORKER_NAME = "updatePusher";

  /** input slot name. */
  private static final String INPUT_SLOT_NAME = "recordsToPush";

  /** (optional) output slot name. */
  private static final String OUTPUT_SLOT_NAME = "pushedRecords";

  /** the indexing job where the records should be pushed to. */
  private static final String JOB_TO_PUSH_TO_PARAM = "jobToPushTo";

  /** client parameters for pushing records to remote bulkbuilder. */
  private static final String CLIENT_PARAMS = "remote";

  /** url path of remote bulkbuilder. */
  private static final String URL_PATH_PARAM = "urlPath";

  /** endpoints of remote bulkbuilder. */
  private static final String ENDPOINTS_PARAM = "endpoints";

  /** Reference to the BulkBuilder. */
  private BulkbuilderService _bulkbuilder;

  /** Log. */
  private final Log _log = LogFactory.getLog(getClass());

  @Override
  public String getName() {
    return WORKER_NAME;
  }

  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    String targetJobName = null;
    RestClient client = null;
    String urlPath = null;
    try {
      final AnyMap taskParams = taskContext.getTaskParameters();
      if (taskParams.containsKey(JOB_TO_PUSH_TO_PARAM)) {
        targetJobName = getRequiredParameter(taskContext, JOB_TO_PUSH_TO_PARAM);
      } else {
        final AnyMap clientParams = taskParams.getMap(CLIENT_PARAMS);
        urlPath = clientParams.getStringValue(URL_PATH_PARAM);
        if (urlPath.isEmpty()) {
          throw new IllegalArgumentException("Invalid parameter '" + URL_PATH_PARAM + "' value.");
        }
        client = createClient(taskContext, clientParams);
      }
      final String jobRunId = getJobRunId(taskContext);
      final DeltaImportStrategy deltaUsage = getDeltaImportStrategy(taskContext);
      if (taskContext.getTask().getProperties().containsKey(Task.PROPERTY_IS_COMPLETING_TASK)) {
        if (deltaUsage.isDeltaDeleteEnabled()) {
          deleteObsoleteRecords(taskContext, targetJobName, jobRunId, client, urlPath);
        } else {
          taskContext.getLog().warn(
            "Got a completion task for a job that has delta-delete disabled. Ignoring this task.");
        }
      } else {
        pushUpdatedRecords(taskContext, targetJobName, jobRunId, deltaUsage, client, urlPath);
      }
    } finally {
      if (client != null) {
        client.shutdown();
      }
    }
  }

  /** create rest client if records are to be processed by remote bulkbuilder. */
  private RestClient createClient(final TaskContext taskContext, final AnyMap clientParams) {
    final AnySeq endpoints = clientParams.getSeq(ENDPOINTS_PARAM);
    final List<String> hostsAndPorts = new ArrayList<String>();
    if (endpoints.isEmpty()) {
      throw new IllegalArgumentException("Invalid '" + ENDPOINTS_PARAM + "' value.");
    }
    for (int i = 0; i < endpoints.size(); ++i) {
      hostsAndPorts.add(endpoints.getStringValue(i));
    }
    return new FailoverRestClient(hostsAndPorts);
  }

  /**
   * peform cleanup task: create delete record in target job for obsolete records, or send delete request to remote REST
   * API.
   */
  private void deleteObsoleteRecords(final TaskContext taskContext, final String targetJobName,
    final String jobRunId, final RestClient client, final String urlPath) throws BulkbuilderException,
    ImportingException {
    final String sourceId = getRequiredParameter(taskContext, ImportingConstants.TASK_PARAM_DATA_SOURCE);
    final String sourceAndShardPrefix = getRequiredParameter(taskContext, DELETE_SHARD_PARAM);
    taskContext.addToCounter("deltaDelete.tasks", 1);
    final Collection<DeltaService.EntryId> entryIds =
      getUnvisitedEntriesTimed(jobRunId, sourceAndShardPrefix, taskContext);
    for (final DeltaService.EntryId entryId : entryIds) {
      if (taskContext.isCanceled()) {
        return;
      }
      taskContext.addToCounter("deltaDelete.unvisitedRecords", 1);
      final Record deleteRecord = DataFactory.DEFAULT.createRecord(entryId.getRecordId(), sourceId);
      if (client != null) {
        try {
          client.delete(urlPath, deleteRecord.getMetadata());
          taskContext.addToCounter("deltaDelete.deletedRecords", 1);
          deleteDeltaEntryTimed(sourceId, entryId, taskContext);
        } catch (final RestException ex) {
          throw new ImportingException("Could not submit delete record '" + entryId + "' to " + urlPath, ex);
        } catch (final IOException ex) {
          throw new ImportingException("Could not submit delete record '" + entryId + "' to " + urlPath, ex, true);
        }
      } else {
        try {
          _bulkbuilder.deleteRecord(targetJobName, deleteRecord);
          taskContext.addToCounter("deltaDelete.deletedRecords", 1);
          deleteDeltaEntryTimed(sourceId, entryId, taskContext);
        } catch (final InvalidRecordException ex) {
          taskContext.getLog().warn("Could not submit delete record '" + entryId + "' to bulkbuilder.", ex);
        } catch (final DeltaException ex) {
          taskContext.getLog().warn("Could not delete entry for record'" + entryId + "' from delta service.", ex);
        }
      }
    }
  }

  /**
   * peform standard task: push updated records from input bulk to target job.
   * 
   * @throws ImportingException
   */
  private void pushUpdatedRecords(final TaskContext taskContext, final String targetJobName, final String jobRunId,
    final DeltaImportStrategy deltaUsage, final RestClient client, final String urlPath)
    throws ObjectStoreException, IOException, BulkbuilderException, ImportingException {
    final Inputs inputs = taskContext.getInputs();
    final RecordInput recordInput = inputs.getAsRecordInput(INPUT_SLOT_NAME);
    final Outputs outputs = taskContext.getOutputs();
    final RecordOutput recordOutput = outputs.getAsRecordOutput(OUTPUT_SLOT_NAME);
    Record record = recordInput.getRecord();
    while (record != null && !taskContext.isCanceled()) {
      // this hash value reflects the changes of the record. It is set if a DeltaIndexing-Worker is called before.
      final String deltaHash = record.getMetadata().getStringValue(ImportingConstants.ATTRIBUTE_DELTA_HASH);
      if (deltaHash != null) {
        // check is done to avoid duplicates, because in case of an error the task may be retried
        if (deltaUsage.isDeltaCheckDisabled()
          || checkDeltaStateTimed(jobRunId, record, deltaHash, taskContext) != State.UPTODATE) {
          pushToBulkbuilder(taskContext, targetJobName, record, recordOutput, client, urlPath);
          if (deltaUsage.isSetStateEnabled()) {
            markAsUpdatedTimed(jobRunId, record, deltaHash, taskContext);
          }
        } else if (_log.isDebugEnabled()) {
          _log.debug("Skipping record '" + record.getId() + "'");
        }
      } else {
        pushToBulkbuilder(taskContext, targetJobName, record, recordOutput, client, urlPath);
      }
      record = recordInput.getRecord();
    }
  }

  /**
   * send record to local bulkbuilder and write it to output bulk, if available, or send record to remote bulkbuilder.
   */
  private void pushToBulkbuilder(final TaskContext taskContext, final String jobName, final Record record,
    final RecordOutput recordOutput, final RestClient client, final String urlPath) throws BulkbuilderException,
    ObjectStoreException, IOException, ImportingException {
    if (_log.isDebugEnabled()) {
      _log.debug("Sending record '" + record.getId() + "'");
    }
    if (client != null) {
      try {
        client.post(urlPath, record);
      } catch (final RestException ex) {
        throw new ImportingException("Could not submit record '" + record.getId() + "' to " + urlPath, ex);
      } catch (final IOException ex) {
        throw new ImportingException("Could not submit record '" + record.getId() + "' to " + urlPath, ex, true);
      }
    } else {
      try {
        _bulkbuilder.addRecord(jobName, record);
        if (recordOutput != null) {
          recordOutput.writeRecord(record);
        }
      } catch (final InvalidRecordException ex) {
        taskContext.getLog().warn("Could not submit record '" + record.getId() + "' to bulkbuilder.", ex);
      }
    }
  }

  /** DS service reference bind method. */
  public void setBulkbuilderService(final BulkbuilderService service) {
    _bulkbuilder = service;
  }

  /** DS service reference unbind method. */
  public void unsetBulkbuilderService(final BulkbuilderService service) {
    if (_bulkbuilder == service) {
      _bulkbuilder = null;
    }
  }
}
