/*********************************************************************************************************************
 * Copyright (c) 2008, 2013 Empolis Information Management GmbH and brox IT Solutions GmbH. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 *********************************************************************************************************************/
package org.eclipse.smila.processing.httphandler;

import java.util.List;
import java.util.UUID;

import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.utils.MaybeRecoverableException;

public abstract class AProcessHandler extends JsonRequestHandler {

  /**
   * blackboard factory.
   */
  private BlackboardFactory _blackboardFactory;

  /** {@inheritDoc} */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    try {
      if (inputRecord == null || inputRecord.getMetadata().isEmpty()) {
        throw new EmptyQueryException("Cannot process an empty record.");
      }
      final String elementName = getElementName(requestUri);
      ensureQueryId(elementName, inputRecord);
      final Blackboard blackboard = _blackboardFactory.createTransientBlackboard();
      blackboard.setRecord(inputRecord);
      final String[] resultIds = doProcess(elementName, blackboard, new String[] { inputRecord.getId() });
      Record result = null;
      if (resultIds != null && resultIds.length > 0) {
        // TODO: we should support returning multiple results some day ...
        // if (resultIds.length == 1) {
        result = blackboard.getRecord(resultIds[0]);
        // } else {
        // final AnySeq results = DataFactory.DEFAULT.createAnySeq();
        // for (final String id : resultIds) {
        // results.add(blackboard.getMetadata(id));
        // }
        // result = DataFactory.DEFAULT.createRecord();
        // result.getMetadata().put("records", results);
        // return result;
        // }
      }
      blackboard.unload();
      return result;
    } catch (final BlackboardAccessException ex) {
      throw new ProcessingException("Blackboard error: ", ex);
    }
  }

  protected abstract String[] doProcess(String elementName, Blackboard blackboard, String[] strings)
    throws BlackboardAccessException, ProcessingException;

  /** get name of processing element to use. */
  protected String getElementName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern " + getUriPattern());
  }

  /** set blackboard factory reference (used by DS). */
  public void setBlackboardFactory(final BlackboardFactory factory) {
    _blackboardFactory = factory;
  }

  /** remove blackboard factory reference (used by DS). */
  public void unsetBlackboardFactory(final BlackboardFactory factory) {
    if (_blackboardFactory == factory) {
      _blackboardFactory = null;
    }
  }

  /**
   * {@inheritDoc}
   * 
   * PipelineHandler supports GET and POST.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "POST".equals(method);
  }

  /**
   * {@inheritDoc}
   * 
   * empty queries return BAD_REQUEST, not INTERNAL_SERVER_ERROR.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof EmptyQueryException) {
      return HttpStatus.BAD_REQUEST;
    } else if (ex instanceof MaybeRecoverableException) {
      if (!((MaybeRecoverableException) ex).isRecoverable()) {
        return HttpStatus.BAD_REQUEST;
      }
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * ensure that the query object has a Id set. If none is provided by the client create one using the pipeline name as
   * source and a random {@link UUID} as key.
   * 
   * @param pipeletName
   *          pipeline name
   * @param query
   *          query record.
   */
  protected void ensureQueryId(final String pipeletName, final Record query) {
    if (query.getId() == null) {
      UUID uuid = null;
      synchronized (this) {
        uuid = UUID.randomUUID();
      }
      query.setId(pipeletName + "-" + uuid.toString());
    }
  }

  /**
   * special exception to denote empty queries ( -> BAD_REQUEST, not SERVER_ERROR).
   * 
   * @author jschumacher
   * 
   */
  protected static class EmptyQueryException extends Exception {
    /**
     * ... is serializable
     */
    private static final long serialVersionUID = 1L;

    /**
     * create exception.
     * 
     * @param message
     *          description.
     */
    public EmptyQueryException(final String message) {
      super(message);
    }
  }

}
