/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Drazen Cindric (Attensity Europe GmbH) - initial API and implementation
 *******************************************************************************/
package org.eclipse.smila.processing.httphandler;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.utils.http.NotFoundHTTPResult;

/**
 * Implements the handling of HTTP requests for reading and deleting a workflow definition.
 * 
 * @author drazen
 * 
 */
public class PipelineHandler extends JsonRequestHandler {

  /** Exception message if no definition has been found. */
  public static final String EXCEPTION_MESSAGE = "There is no definition for given pipeline";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** associated workflow processor. */
  private WorkflowProcessor _processor;

  /**
   * {@inheritDoc}
   * 
   * PipelineHandler supports GET and POST.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "DELETE".equals(method);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final String workflowName = getWorkflowName(requestUri);
    if ("DELETE".equals(method)) {
      if (_log.isInfoEnabled()) {
        _log.info("Deleting pipeline: " + workflowName);
      }
      _processor.deleteWorkflowDefinition(workflowName);
      return null;
    } else {
      final AnyMap definitionAny = _processor.getWorkflowDefinition(workflowName);
      if (definitionAny == null) {
        throw new DefinitionNotFoundException(EXCEPTION_MESSAGE + " -" + workflowName + "-.");
      }
      return definitionAny;
    }
  }

  /**
   * get pipeline name from URI.
   * 
   * @param requestUri
   *          request uri
   * @return index name
   */
  private String getWorkflowName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern " + getUriPattern());
  }

  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof ProcessingException) {
      return ((ProcessingException) ex).isRecoverable() ? HttpStatus.INTERNAL_SERVER_ERROR : HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * set workflow processor reference.
   * 
   * @param processor
   *          workflow processor
   */
  public void setProcessor(final WorkflowProcessor processor) {
    _processor = processor;
  }

  /**
   * remove workflow processor reference.
   * 
   * @param processor
   *          workflow processor
   */
  public void unsetProcessor(final WorkflowProcessor processor) {
    if (_processor == processor) {
      _processor = null;
    }
  }

  /**
   * special exception to denote that for given workflo name no definition has been found.
   * 
   * @author drazen
   * 
   */
  private static class DefinitionNotFoundException extends Exception implements NotFoundHTTPResult {
    /**
     * ... is serializable
     */
    private static final long serialVersionUID = 1L;

    /**
     * create exception.
     * 
     * @param message
     *          description.
     */
    public DefinitionNotFoundException(final String message) {
      super(message);
    }
  }
}
