/**
 *
 */
package org.eclipse.smila.solr.params;

import org.apache.commons.lang.NullArgumentException;
import org.apache.commons.lang.StringUtils;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.search.api.QueryConstants;

/**
 * @author pwissel
 *
 */
public class UpdateParams extends SolrParams {

  public static final String UPDATE = "update";

  public static final String ATTACHMENTS = "attachments";

  public static final String COMMIT_WITHIN_MS = "commitWithinMs";

  public static final int COMMIT_WITHIN_MS_DEFAULT = -1;

  public static final String DOCUMENT_BOOST = "documentBoost";

  public static final float DOCUMENT_BOOST_DEFAULT = 1.0f;

  public static final String MAPPING = "mapping";

  public static final String OPERATION = "operation";

  public static final String DELETE_QUERY = "deleteQuery";

  public enum Operation {
    ADD, DELETE_BY_ID, DELETE_BY_QUERY, NONE;
    public static Operation get(final String label) {
      return valueOf(label.toUpperCase());
    }
  }

  /**
   * @param configuration
   */
  public UpdateParams(AnyMap configuration) {
    super(configuration);
  }

  /**
   * @param blackboard
   */
  public UpdateParams(Blackboard blackboard) {
    super(blackboard);
  }

  /**
   * @param blackboard
   * @param configuration
   */
  public UpdateParams(Blackboard blackboard, AnyMap configuration) {
    super(blackboard, configuration);
  }

  /**
   * @param blackboard
   * @param configuration
   * @param recordId
   */
  public UpdateParams(Blackboard blackboard, AnyMap configuration, String recordId) {
    super(blackboard, configuration, recordId);
  }

  /**
   * @param blackboard
   * @param recordId
   */
  public UpdateParams(Blackboard blackboard, String recordId) {
    super(blackboard, recordId);
  }

  public Boolean getAttachments(final boolean defaultIfNull) {
    final Boolean val = getSolrParamAsBoolean(UPDATE, ATTACHMENTS);
    if (val == null && defaultIfNull) {
      return false;
    }
    return val;
  }

  public String getIndexName() {
    final String parameterAttribute = getParameterAttribute();
    setParameterAttribute(null);
    final String index = getParameter(QueryConstants.INDEXNAME, null);
    setParameterAttribute(parameterAttribute);
    return index;
  }

  public Integer getCommitWithinMs(final boolean defaultIfNull) {
    final Integer val = getSolrParamAsInteger(UPDATE, COMMIT_WITHIN_MS);
    if (val == null && defaultIfNull) {
      return COMMIT_WITHIN_MS_DEFAULT;
    }
    return val;
  }

  public Float getDocumentBoost(final boolean defaultIfNull) {
    final Float val = getSolrParamAsFloat(UPDATE, DOCUMENT_BOOST);
    if (val == null && defaultIfNull) {
      return DOCUMENT_BOOST_DEFAULT;
    }
    return val;
  }

  public AnyMap getMapping(final boolean defaultIfNull) {
    final AnyMap val = getSolrParamAsAnyMap(UPDATE, MAPPING);
    if (val == null && defaultIfNull) {
      return DataFactory.DEFAULT.createAnyMap();
    }
    return val;
  }

  public Operation getOperation() {
    final String val = getSolrParamAsString(UPDATE, OPERATION);
    if (val == null) {
      throw new NullArgumentException(OPERATION);
    }
    try {
      return Operation.get(val);
    } catch (final Exception exception) {
      final String message =
        String.format("Operation must be one of [%s].", StringUtils.join(Operation.values(), ", "));
      throw new IllegalArgumentException(message, exception);
    }
  }

  public String getDeleteQeuery() {
    return getSolrParamAsString(UPDATE, DELETE_QUERY);
  }

}
