/*********************************************************************************************************************
 * Copyright (c) 2008, 2012 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 **********************************************************************************************************************/
package org.eclipse.smila.utils.codec;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;

import org.apache.commons.codec.BinaryDecoder;
import org.apache.commons.codec.BinaryEncoder;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.EncoderException;
import org.apache.commons.codec.binary.Base64InputStream;
import org.apache.commons.io.IOUtils;

/**
 * Utility class for encoding/decoding.
 * 
 * @author stuc07
 * 
 */
public final class EncodingTools {

  /** Private Constructor to avoid instance creation. */
  private EncodingTools() {
  }

  /**
   * Encodes a String with base64.
   * 
   * @param input
   *          the string to encode. The String is expected to be a UTF-8 String.
   * @return the encoded string
   * @throws IOException
   */
  public static String toBase64(final String input) throws IOException {
    return toBase64(input, "UTF-8");
  }

  /**
   * Encodes a String with base64.
   * 
   * @param input
   *          the string to encode.
   * @param charset
   *          charset of the String (e.g. UTF-8)
   * @return the encoded string
   * @throws IOException
   */
  public static String toBase64(final String input, final String charset) throws IOException {
    try (InputStream inputStream = IOUtils.toInputStream(input, charset)) {
      return toBase64(inputStream);
    }
  }

  /**
   * Encodes a byte[] with base64.
   * 
   * @param bytes
   *          the byte[] to encode
   * @return the encoded String
   * @throws IOException
   */
  public static String toBase64(final byte[] bytes) throws IOException {
    try (ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes)) {
      return toBase64(inputStream);
    }
  }

  /**
   * Encodes the data of an InputStream with base64.
   * 
   * @param in
   *          the InputStream to encode
   * @return the encoded String
   * @throws IOException
   */
  public static String toBase64(final InputStream in) throws IOException {
    try (final Base64InputStream encodedStream = new Base64InputStream(in, true, -1, null)) {
      return IOUtils.toString(encodedStream);
    }
  }

  /**
   * Encodes a String using the given encoder.
   * 
   * @param input
   *          the String to encode. The String is expected to be a UTF-8 String.
   * @param encoder
   *          an encoder (see org.apache.commons.codec)
   * @return the encoded String
   * @throws EncoderException
   */
  public static String encode(final String input, final BinaryEncoder encoder) throws EncoderException {
    return encode(input, encoder, Charset.forName("UTF-8"));
  }

  /**
   * Encodes a String using the given encoder.
   * 
   * @param input
   *          input the String to encode
   * @param encoder
   *          an encoder (see org.apache.commons.codec)
   * @param charset
   *          the charset of the string
   * @return the encoded String
   * @throws EncoderException
   */
  public static String encode(final String input, final BinaryEncoder encoder, final Charset charset)
    throws EncoderException {
    return new String(encode(input.getBytes(charset), encoder), charset);
  }

  /**
   * Encodes a byte[] using the given encoder.
   * 
   * @param input
   *          the byte[] to encode
   * @param encoder
   *          an encoder (see org.apache.commons.codec)
   * @return the encoded String
   * @throws EncoderException
   */
  public static byte[] encode(final byte[] input, final BinaryEncoder encoder) throws EncoderException {
    return encoder.encode(input);
  }

  /**
   * Encodes the data of an InputStream a String using the given encoder.
   * 
   * @param input
   *          the InputStream to encode
   * @return the encoded String
   * @throws IOException
   */
  public static String base64toString(final String input) throws IOException {
    return base64toString(input, "UTF-8");
  }

  /**
   * Decodes a base64 encoded String.
   * 
   * @param input
   *          the base64 encoded String
   * @param charset
   *          charset of the String
   * @return
   * @throws IOException
   */
  public static String base64toString(final String input, final String charset) throws IOException {
    try (final InputStream inputStream = base64toStream(input, charset)) {
      return IOUtils.toString(inputStream);
    }
  }

  /**
   * Decodes a base64 encoded String.
   * 
   * @param input
   * @return the decoded String
   * @throws IOException
   */
  public static byte[] base64toByteArray(final String input) throws IOException {
    return base64toByteArray(input, "UTF-8");
  }

  /**
   * Decodes a base64 encoded String into a byte[].
   * 
   * @param input
   *          the base64 encoded String
   * @param charset
   *          charset of the string
   * @return the decoded byte[]
   * @throws IOException
   */
  public static byte[] base64toByteArray(final String input, final String charset) throws IOException {
    try (final InputStream inputStream = base64toStream(input, charset)) {
      return IOUtils.toByteArray(inputStream);
    }
  }

  /**
   * Decodes a base64 encoded String into an InputStream.
   * 
   * @param input
   *          the base64 encoded String
   * @return the decoded InputStream
   * @throws IOException
   */
  public static InputStream base64toStream(final String input) throws IOException {
    return base64toStream(input, "UTF-8");
  }

  /**
   * Decodes a base64 encoded String into an InputStream.
   * 
   * @param input
   *          the base64 encoded String
   * @param charset
   *          the charset of the String
   * @return the decoded InputStream
   * @throws IOException
   */
  public static InputStream base64toStream(final String input, final String charset) throws IOException {
    try (final InputStream inputStream = IOUtils.toInputStream(input, charset)) {
      return new Base64InputStream(inputStream, false);
    }
  }

  /**
   * Decodes a String using a decoder.
   * 
   * @param input
   *          the String to decode
   * @param decoder
   *          a decoder (see org.apache.commons.codec)
   * @return the decoded String
   * @throws DecoderException
   */
  public static String decode(final String input, final BinaryDecoder decoder) throws DecoderException {
    return decode(input, decoder, Charset.forName("UTF-8"));
  }

  /**
   * Decodes a String using a decoder.
   * 
   * @param input
   *          the String to decode
   * @param decoder
   *          a decoder (see org.apache.commons.codec)
   * @param charset
   *          the charset of the String
   * @return the decoded String
   * @throws DecoderException
   */
  public static String decode(final String input, final BinaryDecoder decoder, final Charset charset)
    throws DecoderException {
    return new String(decode(input.getBytes(), decoder), charset);
  }

  /**
   * Decodes a String using a decoder into a byte[9.
   * 
   * @param input
   *          the String to decode
   * @param decoder
   *          decoder a decoder (see org.apache.commons.codec)
   * @return the decoded byte[]
   * @throws DecoderException
   */
  public static byte[] decode(final byte[] input, final BinaryDecoder decoder) throws DecoderException {
    return decoder.decode(input);
  }
}
